package protocol

import (
	"crypto/sha256"
	"testing"

	"github.com/stretchr/testify/assert"
	"go.uber.org/mock/gomock"

	"github.com/go-webauthn/webauthn/metadata"
	"github.com/go-webauthn/webauthn/testing/mocks"
)

func TestVerifyAndroidKeyFormat(t *testing.T) {
	type args struct {
		att            AttestationObject
		clientDataHash []byte
	}

	successAttResponse0 := attestationTestUnpackResponse(t, androidKeyTestResponse0["success"]).Response.AttestationObject
	successClientDataHash0 := sha256.Sum256(attestationTestUnpackResponse(t, androidKeyTestResponse0["success"]).Raw.AttestationResponse.ClientDataJSON)
	successAttResponse1 := attestationTestUnpackResponse(t, androidKeyTestResponse1["success"]).Response.AttestationObject
	successClientDataHash1 := sha256.Sum256(attestationTestUnpackResponse(t, androidKeyTestResponse1["success"]).Raw.AttestationResponse.ClientDataJSON)
	successAttResponse2 := attestationTestUnpackResponse(t, androidKeyTestResponse2["success"]).Response.AttestationObject
	successClientDataHash2 := sha256.Sum256(attestationTestUnpackResponse(t, androidKeyTestResponse2["success"]).Raw.AttestationResponse.ClientDataJSON)

	testCases := []struct {
		name     string
		setup    func(t *testing.T, mds *mocks.MockMetadataProvider)
		mds      bool
		args     args
		expected string
		x5cs     []any
		err      string
	}{
		{
			"ShouldHandleSuccessfulAttestationResponse0",
			nil,
			false,
			args{
				successAttResponse0,
				successClientDataHash0[:],
			},
			string(metadata.BasicFull),
			[]any{[]byte{0x30, 0x82, 0x3, 0x1, 0x30, 0x82, 0x2, 0xa7, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0x1, 0x1, 0x30, 0xa, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x4, 0x3, 0x2, 0x30, 0x81, 0xce, 0x31, 0x45, 0x30, 0x43, 0x6, 0x3, 0x55, 0x4, 0x3, 0xc, 0x3c, 0x46, 0x41, 0x4b, 0x45, 0x20, 0x41, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x20, 0x4b, 0x65, 0x79, 0x73, 0x74, 0x6f, 0x72, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x41, 0x74, 0x74, 0x65, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x74, 0x65, 0x20, 0x46, 0x41, 0x4b, 0x45, 0x31, 0x31, 0x30, 0x2f, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x1, 0x16, 0x22, 0x63, 0x6f, 0x6e, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x6e, 0x63, 0x65, 0x2d, 0x74, 0x6f, 0x6f, 0x6c, 0x73, 0x40, 0x66, 0x69, 0x64, 0x6f, 0x61, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x16, 0x30, 0x14, 0x6, 0x3, 0x55, 0x4, 0xa, 0xc, 0xd, 0x46, 0x49, 0x44, 0x4f, 0x20, 0x41, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x31, 0xc, 0x30, 0xa, 0x6, 0x3, 0x55, 0x4, 0xb, 0xc, 0x3, 0x43, 0x57, 0x47, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x55, 0x53, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x8, 0xc, 0x2, 0x4d, 0x59, 0x31, 0x12, 0x30, 0x10, 0x6, 0x3, 0x55, 0x4, 0x7, 0xc, 0x9, 0x57, 0x61, 0x6b, 0x65, 0x66, 0x69, 0x65, 0x6c, 0x64, 0x30, 0x20, 0x17, 0xd, 0x37, 0x30, 0x30, 0x32, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x18, 0xf, 0x32, 0x30, 0x39, 0x39, 0x30, 0x31, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x29, 0x31, 0x27, 0x30, 0x25, 0x6, 0x3, 0x55, 0x4, 0x3, 0xc, 0x1e, 0x46, 0x41, 0x4b, 0x45, 0x20, 0x41, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x20, 0x4b, 0x65, 0x79, 0x73, 0x74, 0x6f, 0x72, 0x65, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x46, 0x41, 0x4b, 0x45, 0x30, 0x59, 0x30, 0x13, 0x6, 0x7, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x2, 0x1, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x3, 0x1, 0x7, 0x3, 0x42, 0x0, 0x4, 0x1b, 0x87, 0xe0, 0x50, 0x4, 0x9c, 0xfb, 0x25, 0xe4, 0x36, 0xed, 0xd5, 0x6f, 0xbb, 0x7b, 0x72, 0x46, 0x27, 0xa2, 0x11, 0xe5, 0xf2, 0xd, 0x8a, 0x1a, 0x59, 0xab, 0x51, 0x77, 0x2f, 0xc3, 0xee, 0xad, 0xd3, 0x2b, 0xde, 0xa3, 0xba, 0x62, 0xa5, 0xc0, 0x89, 0x79, 0x64, 0xd, 0xb3, 0x1e, 0x62, 0x8d, 0x2f, 0x3e, 0xa, 0xdb, 0x3b, 0x0, 0x36, 0xad, 0x57, 0xe2, 0xd1, 0xe9, 0xf0, 0xc3, 0x59, 0xa3, 0x82, 0x1, 0x16, 0x30, 0x82, 0x1, 0x12, 0x30, 0xb, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x4, 0x4, 0x3, 0x2, 0x7, 0x80, 0x30, 0x81, 0xe1, 0x6, 0xa, 0x2b, 0x6, 0x1, 0x4, 0x1, 0xd6, 0x79, 0x2, 0x1, 0x11, 0x4, 0x81, 0xd2, 0x30, 0x81, 0xcf, 0x2, 0x1, 0x2, 0xa, 0x1, 0x0, 0x2, 0x1, 0x1, 0xa, 0x1, 0x0, 0x4, 0x20, 0xdc, 0xd1, 0x4a, 0x17, 0xb5, 0x4d, 0x42, 0xc3, 0x2, 0x2d, 0x5b, 0x79, 0xde, 0xd9, 0xf6, 0x8a, 0xd, 0xc1, 0x42, 0x69, 0xb1, 0x48, 0xdf, 0x50, 0x6e, 0x7d, 0x71, 0x5, 0x54, 0xaf, 0xc4, 0x37, 0x4, 0x0, 0x30, 0x69, 0xbf, 0x85, 0x3d, 0x8, 0x2, 0x6, 0x1, 0x5e, 0xd3, 0xe3, 0xcf, 0xa0, 0xbf, 0x85, 0x45, 0x59, 0x4, 0x57, 0x30, 0x55, 0x31, 0x2f, 0x30, 0x2d, 0x4, 0x28, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x2e, 0x6b, 0x65, 0x79, 0x73, 0x74, 0x6f, 0x72, 0x65, 0x2e, 0x61, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x6b, 0x65, 0x79, 0x73, 0x74, 0x6f, 0x72, 0x65, 0x64, 0x65, 0x6d, 0x6f, 0x2, 0x1, 0x1, 0x31, 0x22, 0x4, 0x20, 0x74, 0xcf, 0xcb, 0x50, 0x74, 0x88, 0xf5, 0x29, 0x10, 0x85, 0x91, 0xc7, 0xa5, 0x5, 0x91, 0x9f, 0x32, 0x77, 0x32, 0xfb, 0xc1, 0xd8, 0x3, 0x52, 0x6a, 0xea, 0x98, 0x0, 0x6, 0xd2, 0xd8, 0x98, 0x30, 0x32, 0xa1, 0x5, 0x31, 0x3, 0x2, 0x1, 0x2, 0xa2, 0x3, 0x2, 0x1, 0x3, 0xa3, 0x4, 0x2, 0x2, 0x1, 0x0, 0xa5, 0x5, 0x31, 0x3, 0x2, 0x1, 0x4, 0xaa, 0x3, 0x2, 0x1, 0x1, 0xbf, 0x83, 0x78, 0x3, 0x2, 0x1, 0x2, 0xbf, 0x85, 0x3e, 0x3, 0x2, 0x1, 0x0, 0xbf, 0x85, 0x3f, 0x2, 0x5, 0x0, 0x30, 0x1f, 0x6, 0x3, 0x55, 0x1d, 0x23, 0x4, 0x18, 0x30, 0x16, 0x80, 0x14, 0x52, 0x9a, 0x1b, 0x32, 0xe0, 0x56, 0xaa, 0xcd, 0x51, 0xf, 0xef, 0x5f, 0x80, 0x37, 0x6, 0x8b, 0xf2, 0x91, 0x23, 0x1d, 0x30, 0xa, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x4, 0x3, 0x2, 0x3, 0x48, 0x0, 0x30, 0x45, 0x2, 0x21, 0x0, 0xc0, 0x3d, 0x5e, 0xde, 0x40, 0x85, 0x9f, 0x2f, 0x90, 0x42, 0x9a, 0x3f, 0x36, 0x36, 0xcc, 0xc3, 0x95, 0xd, 0x88, 0x26, 0xc3, 0x99, 0xab, 0xf5, 0xb1, 0x89, 0xdc, 0x31, 0xf, 0xb3, 0xa7, 0xfd, 0x2, 0x20, 0x16, 0x2a, 0x69, 0x96, 0x5, 0x5, 0xe8, 0xe1, 0xef, 0x98, 0x78, 0x83, 0x75, 0xa1, 0xc9, 0x7c, 0x5c, 0x1e, 0x7a, 0x8e, 0xb4, 0x3a, 0xd0, 0x83, 0x18, 0x37, 0xed, 0x8d, 0x1, 0x26, 0x17, 0xb1}, []byte{0x30, 0x82, 0x2, 0xea, 0x30, 0x82, 0x2, 0x91, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0x1, 0x2, 0x30, 0xa, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x4, 0x3, 0x2, 0x30, 0x81, 0xc6, 0x31, 0x3d, 0x30, 0x3b, 0x6, 0x3, 0x55, 0x4, 0x3, 0xc, 0x34, 0x46, 0x41, 0x4b, 0x45, 0x20, 0x41, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x20, 0x4b, 0x65, 0x79, 0x73, 0x74, 0x6f, 0x72, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x41, 0x74, 0x74, 0x65, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x52, 0x6f, 0x6f, 0x74, 0x20, 0x46, 0x41, 0x4b, 0x45, 0x31, 0x31, 0x30, 0x2f, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x1, 0x16, 0x22, 0x63, 0x6f, 0x6e, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x6e, 0x63, 0x65, 0x2d, 0x74, 0x6f, 0x6f, 0x6c, 0x73, 0x40, 0x66, 0x69, 0x64, 0x6f, 0x61, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x16, 0x30, 0x14, 0x6, 0x3, 0x55, 0x4, 0xa, 0xc, 0xd, 0x46, 0x49, 0x44, 0x4f, 0x20, 0x41, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x31, 0xc, 0x30, 0xa, 0x6, 0x3, 0x55, 0x4, 0xb, 0xc, 0x3, 0x43, 0x57, 0x47, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x55, 0x53, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x8, 0xc, 0x2, 0x4d, 0x59, 0x31, 0x12, 0x30, 0x10, 0x6, 0x3, 0x55, 0x4, 0x7, 0xc, 0x9, 0x57, 0x61, 0x6b, 0x65, 0x66, 0x69, 0x65, 0x6c, 0x64, 0x30, 0x1e, 0x17, 0xd, 0x31, 0x38, 0x30, 0x35, 0x30, 0x39, 0x31, 0x32, 0x33, 0x31, 0x34, 0x34, 0x5a, 0x17, 0xd, 0x34, 0x35, 0x30, 0x39, 0x32, 0x34, 0x31, 0x32, 0x33, 0x31, 0x34, 0x34, 0x5a, 0x30, 0x81, 0xce, 0x31, 0x45, 0x30, 0x43, 0x6, 0x3, 0x55, 0x4, 0x3, 0xc, 0x3c, 0x46, 0x41, 0x4b, 0x45, 0x20, 0x41, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x20, 0x4b, 0x65, 0x79, 0x73, 0x74, 0x6f, 0x72, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x41, 0x74, 0x74, 0x65, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x74, 0x65, 0x20, 0x46, 0x41, 0x4b, 0x45, 0x31, 0x31, 0x30, 0x2f, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x1, 0x16, 0x22, 0x63, 0x6f, 0x6e, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x6e, 0x63, 0x65, 0x2d, 0x74, 0x6f, 0x6f, 0x6c, 0x73, 0x40, 0x66, 0x69, 0x64, 0x6f, 0x61, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x16, 0x30, 0x14, 0x6, 0x3, 0x55, 0x4, 0xa, 0xc, 0xd, 0x46, 0x49, 0x44, 0x4f, 0x20, 0x41, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x31, 0xc, 0x30, 0xa, 0x6, 0x3, 0x55, 0x4, 0xb, 0xc, 0x3, 0x43, 0x57, 0x47, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x55, 0x53, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x8, 0xc, 0x2, 0x4d, 0x59, 0x31, 0x12, 0x30, 0x10, 0x6, 0x3, 0x55, 0x4, 0x7, 0xc, 0x9, 0x57, 0x61, 0x6b, 0x65, 0x66, 0x69, 0x65, 0x6c, 0x64, 0x30, 0x59, 0x30, 0x13, 0x6, 0x7, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x2, 0x1, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x3, 0x1, 0x7, 0x3, 0x42, 0x0, 0x4, 0xab, 0x50, 0x61, 0x2b, 0x62, 0x4d, 0x17, 0xbb, 0xc3, 0xb5, 0x1b, 0x4, 0x49, 0x3b, 0x5, 0x49, 0xb, 0x8c, 0x1f, 0x8b, 0x9d, 0xbc, 0xf3, 0x79, 0xb2, 0xde, 0xdc, 0xa0, 0x7c, 0x52, 0xa1, 0xd7, 0xb6, 0xd, 0xd3, 0x21, 0x75, 0xdc, 0x2b, 0x7e, 0xdd, 0xbc, 0x26, 0xdb, 0x8, 0xd5, 0x5f, 0x86, 0xf6, 0x71, 0xb6, 0x94, 0xdc, 0x30, 0x4d, 0xa4, 0xa2, 0x8a, 0x8c, 0x4e, 0x3, 0xa6, 0xf5, 0xc7, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x12, 0x6, 0x3, 0x55, 0x1d, 0x13, 0x1, 0x1, 0xff, 0x4, 0x8, 0x30, 0x6, 0x1, 0x1, 0xff, 0x2, 0x1, 0x0, 0x30, 0xe, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x1, 0x1, 0xff, 0x4, 0x4, 0x3, 0x2, 0x2, 0x84, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x1d, 0xe, 0x4, 0x16, 0x4, 0x14, 0xa3, 0xd2, 0xaa, 0x2c, 0xef, 0xd, 0x8c, 0xf2, 0x24, 0x2, 0xd5, 0x1c, 0xb4, 0x60, 0xbc, 0xbf, 0x6a, 0x5b, 0x24, 0x14, 0x30, 0x1f, 0x6, 0x3, 0x55, 0x1d, 0x23, 0x4, 0x18, 0x30, 0x16, 0x80, 0x14, 0x52, 0x9a, 0x1b, 0x32, 0xe0, 0x56, 0xaa, 0xcd, 0x51, 0xf, 0xef, 0x5f, 0x80, 0x37, 0x6, 0x8b, 0xf2, 0x91, 0x23, 0x1d, 0x30, 0xa, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x4, 0x3, 0x2, 0x3, 0x47, 0x0, 0x30, 0x44, 0x2, 0x20, 0x69, 0xdd, 0x9e, 0xa3, 0xf1, 0x81, 0xfb, 0x42, 0x4a, 0x39, 0xad, 0x1a, 0xa, 0xdf, 0xb9, 0xd2, 0xe3, 0x33, 0xd7, 0x86, 0xfe, 0xb9, 0x45, 0x60, 0x55, 0xfa, 0x3d, 0xcc, 0x99, 0x78, 0xb9, 0xd2, 0x2, 0x20, 0x3e, 0xd3, 0xf, 0xa6, 0xf7, 0x6, 0xb5, 0x56, 0xe9, 0x14, 0x24, 0x2a, 0xb0, 0x66, 0xf2, 0x87, 0x13, 0xdd, 0x92, 0x38, 0xbc, 0x96, 0xd7, 0x18, 0x5d, 0x5, 0x5f, 0x32, 0x17, 0x54, 0xa5, 0x16}},
			"",
		},
		{
			"ShouldHandleSuccessfulAttestationResponse1",
			nil,
			false,
			args{
				successAttResponse1,
				successClientDataHash1[:],
			},
			string(metadata.BasicFull),
			[]any{[]byte{0x30, 0x82, 0x3, 0x16, 0x30, 0x82, 0x2, 0xbd, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0x1, 0x1, 0x30, 0xa, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x4, 0x3, 0x2, 0x30, 0x81, 0xe4, 0x31, 0x45, 0x30, 0x43, 0x6, 0x3, 0x55, 0x4, 0x3, 0xc, 0x3c, 0x46, 0x41, 0x4b, 0x45, 0x20, 0x41, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x20, 0x4b, 0x65, 0x79, 0x73, 0x74, 0x6f, 0x72, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x41, 0x74, 0x74, 0x65, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x74, 0x65, 0x20, 0x46, 0x41, 0x4b, 0x45, 0x31, 0x31, 0x30, 0x2f, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x1, 0x16, 0x22, 0x63, 0x6f, 0x6e, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x6e, 0x63, 0x65, 0x2d, 0x74, 0x6f, 0x6f, 0x6c, 0x73, 0x40, 0x66, 0x69, 0x64, 0x6f, 0x61, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x16, 0x30, 0x14, 0x6, 0x3, 0x55, 0x4, 0xa, 0xc, 0xd, 0x46, 0x49, 0x44, 0x4f, 0x20, 0x41, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x31, 0x22, 0x30, 0x20, 0x6, 0x3, 0x55, 0x4, 0xb, 0xc, 0x19, 0x41, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74, 0x69, 0x63, 0x61, 0x74, 0x6f, 0x72, 0x20, 0x41, 0x74, 0x74, 0x65, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x55, 0x53, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x8, 0xc, 0x2, 0x4d, 0x59, 0x31, 0x12, 0x30, 0x10, 0x6, 0x3, 0x55, 0x4, 0x7, 0xc, 0x9, 0x57, 0x61, 0x6b, 0x65, 0x66, 0x69, 0x65, 0x6c, 0x64, 0x30, 0x20, 0x17, 0xd, 0x37, 0x30, 0x30, 0x32, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x18, 0xf, 0x32, 0x30, 0x39, 0x39, 0x30, 0x31, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5a, 0x30, 0x29, 0x31, 0x27, 0x30, 0x25, 0x6, 0x3, 0x55, 0x4, 0x3, 0xc, 0x1e, 0x46, 0x41, 0x4b, 0x45, 0x20, 0x41, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x20, 0x4b, 0x65, 0x79, 0x73, 0x74, 0x6f, 0x72, 0x65, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x46, 0x41, 0x4b, 0x45, 0x30, 0x59, 0x30, 0x13, 0x6, 0x7, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x2, 0x1, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x3, 0x1, 0x7, 0x3, 0x42, 0x0, 0x4, 0x6e, 0xa3, 0x8, 0x12, 0xbb, 0x90, 0x28, 0x46, 0x3f, 0x15, 0x8b, 0xf6, 0x4d, 0x48, 0x50, 0x5b, 0x19, 0x46, 0x49, 0x5c, 0x51, 0xbd, 0x32, 0x40, 0xd3, 0xea, 0x30, 0x3, 0x96, 0xbe, 0xed, 0x38, 0xaf, 0x4d, 0x12, 0x8, 0xd5, 0x5f, 0x9d, 0xfe, 0x7a, 0xb8, 0x17, 0xa7, 0x23, 0xda, 0x15, 0x9a, 0xc0, 0x47, 0xda, 0x72, 0xfb, 0x19, 0xf3, 0x7e, 0x32, 0xb, 0x9f, 0xf2, 0xb3, 0x6f, 0xc2, 0x2a, 0xa3, 0x82, 0x1, 0x16, 0x30, 0x82, 0x1, 0x12, 0x30, 0xb, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x4, 0x4, 0x3, 0x2, 0x7, 0x80, 0x30, 0x81, 0xe1, 0x6, 0xa, 0x2b, 0x6, 0x1, 0x4, 0x1, 0xd6, 0x79, 0x2, 0x1, 0x11, 0x4, 0x81, 0xd2, 0x30, 0x81, 0xcf, 0x2, 0x1, 0x2, 0xa, 0x1, 0x0, 0x2, 0x1, 0x1, 0xa, 0x1, 0x0, 0x4, 0x20, 0x9f, 0x54, 0x49, 0x7c, 0xde, 0x94, 0x83, 0x49, 0xea, 0xe4, 0xf4, 0x8d, 0xe9, 0x70, 0x80, 0x8d, 0x4d, 0xdc, 0xdc, 0xe4, 0xdd, 0xee, 0xe2, 0x3b, 0x76, 0xd5, 0xc5, 0xdd, 0xcc, 0x1b, 0x89, 0x8e, 0x4, 0x0, 0x30, 0x69, 0xbf, 0x85, 0x3d, 0x8, 0x2, 0x6, 0x1, 0x5e, 0xd3, 0xe3, 0xcf, 0xa0, 0xbf, 0x85, 0x45, 0x59, 0x4, 0x57, 0x30, 0x55, 0x31, 0x2f, 0x30, 0x2d, 0x4, 0x28, 0x63, 0x6f, 0x6d, 0x2e, 0x61, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x2e, 0x6b, 0x65, 0x79, 0x73, 0x74, 0x6f, 0x72, 0x65, 0x2e, 0x61, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x6b, 0x65, 0x79, 0x73, 0x74, 0x6f, 0x72, 0x65, 0x64, 0x65, 0x6d, 0x6f, 0x2, 0x1, 0x1, 0x31, 0x22, 0x4, 0x20, 0x74, 0xcf, 0xcb, 0x50, 0x74, 0x88, 0xf5, 0x29, 0x10, 0x85, 0x91, 0xc7, 0xa5, 0x5, 0x91, 0x9f, 0x32, 0x77, 0x32, 0xfb, 0xc1, 0xd8, 0x3, 0x52, 0x6a, 0xea, 0x98, 0x0, 0x6, 0xd2, 0xd8, 0x98, 0x30, 0x32, 0xa1, 0x5, 0x31, 0x3, 0x2, 0x1, 0x2, 0xa2, 0x3, 0x2, 0x1, 0x3, 0xa3, 0x4, 0x2, 0x2, 0x1, 0x0, 0xa5, 0x5, 0x31, 0x3, 0x2, 0x1, 0x4, 0xaa, 0x3, 0x2, 0x1, 0x1, 0xbf, 0x83, 0x78, 0x3, 0x2, 0x1, 0x2, 0xbf, 0x85, 0x3e, 0x3, 0x2, 0x1, 0x0, 0xbf, 0x85, 0x3f, 0x2, 0x5, 0x0, 0x30, 0x1f, 0x6, 0x3, 0x55, 0x1d, 0x23, 0x4, 0x18, 0x30, 0x16, 0x80, 0x14, 0xa3, 0xd2, 0xaa, 0x2c, 0xef, 0xd, 0x8c, 0xf2, 0x24, 0x2, 0xd5, 0x1c, 0xb4, 0x60, 0xbc, 0xbf, 0x6a, 0x5b, 0x24, 0x14, 0x30, 0xa, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x4, 0x3, 0x2, 0x3, 0x47, 0x0, 0x30, 0x44, 0x2, 0x20, 0x2e, 0xd6, 0x1d, 0xbd, 0x8f, 0xc9, 0x42, 0xe7, 0x8e, 0x9d, 0x4c, 0xd5, 0x35, 0x48, 0xdd, 0x6b, 0x62, 0xfd, 0xe0, 0xe1, 0x7b, 0xab, 0x5, 0xc1, 0xeb, 0x90, 0x1, 0xd7, 0x9c, 0x5e, 0xd1, 0xe1, 0x2, 0x20, 0x12, 0xfd, 0xe6, 0x64, 0x16, 0xcc, 0x7f, 0x54, 0xdb, 0x2b, 0x4e, 0xed, 0xd7, 0x7d, 0x9d, 0x83, 0xef, 0x2, 0x2d, 0xfa, 0xf4, 0xfa, 0x9d, 0x39, 0x63, 0x7c, 0x76, 0x78, 0xce, 0xa5, 0x39, 0xc6}, []byte{0x30, 0x82, 0x3, 0x14, 0x30, 0x82, 0x2, 0xba, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0x1, 0x2, 0x30, 0xa, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x4, 0x3, 0x2, 0x30, 0x81, 0xdc, 0x31, 0x3d, 0x30, 0x3b, 0x6, 0x3, 0x55, 0x4, 0x3, 0xc, 0x34, 0x46, 0x41, 0x4b, 0x45, 0x20, 0x41, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x20, 0x4b, 0x65, 0x79, 0x73, 0x74, 0x6f, 0x72, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x41, 0x74, 0x74, 0x65, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x52, 0x6f, 0x6f, 0x74, 0x20, 0x46, 0x41, 0x4b, 0x45, 0x31, 0x31, 0x30, 0x2f, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x1, 0x16, 0x22, 0x63, 0x6f, 0x6e, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x6e, 0x63, 0x65, 0x2d, 0x74, 0x6f, 0x6f, 0x6c, 0x73, 0x40, 0x66, 0x69, 0x64, 0x6f, 0x61, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x16, 0x30, 0x14, 0x6, 0x3, 0x55, 0x4, 0xa, 0xc, 0xd, 0x46, 0x49, 0x44, 0x4f, 0x20, 0x41, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x31, 0x22, 0x30, 0x20, 0x6, 0x3, 0x55, 0x4, 0xb, 0xc, 0x19, 0x41, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74, 0x69, 0x63, 0x61, 0x74, 0x6f, 0x72, 0x20, 0x41, 0x74, 0x74, 0x65, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x55, 0x53, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x8, 0xc, 0x2, 0x4d, 0x59, 0x31, 0x12, 0x30, 0x10, 0x6, 0x3, 0x55, 0x4, 0x7, 0xc, 0x9, 0x57, 0x61, 0x6b, 0x65, 0x66, 0x69, 0x65, 0x6c, 0x64, 0x30, 0x1e, 0x17, 0xd, 0x31, 0x39, 0x30, 0x34, 0x32, 0x35, 0x30, 0x35, 0x34, 0x39, 0x33, 0x32, 0x5a, 0x17, 0xd, 0x34, 0x36, 0x30, 0x39, 0x31, 0x30, 0x30, 0x35, 0x34, 0x39, 0x33, 0x32, 0x5a, 0x30, 0x81, 0xe4, 0x31, 0x45, 0x30, 0x43, 0x6, 0x3, 0x55, 0x4, 0x3, 0xc, 0x3c, 0x46, 0x41, 0x4b, 0x45, 0x20, 0x41, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x20, 0x4b, 0x65, 0x79, 0x73, 0x74, 0x6f, 0x72, 0x65, 0x20, 0x53, 0x6f, 0x66, 0x74, 0x77, 0x61, 0x72, 0x65, 0x20, 0x41, 0x74, 0x74, 0x65, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x20, 0x49, 0x6e, 0x74, 0x65, 0x72, 0x6d, 0x65, 0x64, 0x69, 0x61, 0x74, 0x65, 0x20, 0x46, 0x41, 0x4b, 0x45, 0x31, 0x31, 0x30, 0x2f, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x9, 0x1, 0x16, 0x22, 0x63, 0x6f, 0x6e, 0x66, 0x6f, 0x72, 0x6d, 0x61, 0x6e, 0x63, 0x65, 0x2d, 0x74, 0x6f, 0x6f, 0x6c, 0x73, 0x40, 0x66, 0x69, 0x64, 0x6f, 0x61, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x2e, 0x6f, 0x72, 0x67, 0x31, 0x16, 0x30, 0x14, 0x6, 0x3, 0x55, 0x4, 0xa, 0xc, 0xd, 0x46, 0x49, 0x44, 0x4f, 0x20, 0x41, 0x6c, 0x6c, 0x69, 0x61, 0x6e, 0x63, 0x65, 0x31, 0x22, 0x30, 0x20, 0x6, 0x3, 0x55, 0x4, 0xb, 0xc, 0x19, 0x41, 0x75, 0x74, 0x68, 0x65, 0x6e, 0x74, 0x69, 0x63, 0x61, 0x74, 0x6f, 0x72, 0x20, 0x41, 0x74, 0x74, 0x65, 0x73, 0x74, 0x61, 0x74, 0x69, 0x6f, 0x6e, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x6, 0x13, 0x2, 0x55, 0x53, 0x31, 0xb, 0x30, 0x9, 0x6, 0x3, 0x55, 0x4, 0x8, 0xc, 0x2, 0x4d, 0x59, 0x31, 0x12, 0x30, 0x10, 0x6, 0x3, 0x55, 0x4, 0x7, 0xc, 0x9, 0x57, 0x61, 0x6b, 0x65, 0x66, 0x69, 0x65, 0x6c, 0x64, 0x30, 0x59, 0x30, 0x13, 0x6, 0x7, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x2, 0x1, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x3, 0x1, 0x7, 0x3, 0x42, 0x0, 0x4, 0xab, 0x50, 0x61, 0x2b, 0x62, 0x4d, 0x17, 0xbb, 0xc3, 0xb5, 0x1b, 0x4, 0x49, 0x3b, 0x5, 0x49, 0xb, 0x8c, 0x1f, 0x8b, 0x9d, 0xbc, 0xf3, 0x79, 0xb2, 0xde, 0xdc, 0xa0, 0x7c, 0x52, 0xa1, 0xd7, 0xb6, 0xd, 0xd3, 0x21, 0x75, 0xdc, 0x2b, 0x7e, 0xdd, 0xbc, 0x26, 0xdb, 0x8, 0xd5, 0x5f, 0x86, 0xf6, 0x71, 0xb6, 0x94, 0xdc, 0x30, 0x4d, 0xa4, 0xa2, 0x8a, 0x8c, 0x4e, 0x3, 0xa6, 0xf5, 0xc7, 0xa3, 0x63, 0x30, 0x61, 0x30, 0xf, 0x6, 0x3, 0x55, 0x1d, 0x13, 0x1, 0x1, 0xff, 0x4, 0x5, 0x30, 0x3, 0x1, 0x1, 0xff, 0x30, 0xe, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x1, 0x1, 0xff, 0x4, 0x4, 0x3, 0x2, 0x2, 0x84, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x1d, 0xe, 0x4, 0x16, 0x4, 0x14, 0xa3, 0xd2, 0xaa, 0x2c, 0xef, 0xd, 0x8c, 0xf2, 0x24, 0x2, 0xd5, 0x1c, 0xb4, 0x60, 0xbc, 0xbf, 0x6a, 0x5b, 0x24, 0x14, 0x30, 0x1f, 0x6, 0x3, 0x55, 0x1d, 0x23, 0x4, 0x18, 0x30, 0x16, 0x80, 0x14, 0x52, 0x9a, 0x1b, 0x32, 0xe0, 0x56, 0xaa, 0xcd, 0x51, 0xf, 0xef, 0x5f, 0x80, 0x37, 0x6, 0x8b, 0xf2, 0x91, 0x23, 0x1d, 0x30, 0xa, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x4, 0x3, 0x2, 0x3, 0x48, 0x0, 0x30, 0x45, 0x2, 0x21, 0x0, 0xb1, 0x6f, 0x2e, 0x40, 0x2f, 0xb4, 0x7a, 0xce, 0x6d, 0x39, 0x8d, 0xf0, 0xfd, 0x3e, 0xc8, 0xb2, 0x13, 0xe3, 0xde, 0x8f, 0xff, 0x7, 0x92, 0x10, 0x46, 0xc1, 0xea, 0xf0, 0x89, 0x59, 0x28, 0x7d, 0x2, 0x20, 0x6e, 0xf9, 0x43, 0xa8, 0xac, 0x27, 0x8b, 0x70, 0x5e, 0x16, 0x8b, 0x13, 0x86, 0xf4, 0x4e, 0x4b, 0x8b, 0x69, 0xd7, 0x7b, 0x8b, 0x8c, 0xda, 0xae, 0xa6, 0x6c, 0xa1, 0xa4, 0x96, 0x5f, 0x2e, 0xf9}},
			"",
		},
		{
			"ShouldHandleSuccessfulAttestationResponse2",
			nil,
			false,
			args{
				successAttResponse2,
				successClientDataHash2[:],
			},
			string(metadata.BasicFull),
			[]any{[]byte{0x30, 0x82, 0x2, 0xcc, 0x30, 0x82, 0x2, 0x72, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0x1, 0x1, 0x30, 0xa, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x4, 0x3, 0x2, 0x30, 0x39, 0x31, 0x29, 0x30, 0x27, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x20, 0x64, 0x36, 0x30, 0x32, 0x61, 0x30, 0x33, 0x61, 0x36, 0x37, 0x32, 0x64, 0x38, 0x36, 0x35, 0x62, 0x61, 0x35, 0x61, 0x34, 0x38, 0x35, 0x65, 0x33, 0x33, 0x61, 0x32, 0x30, 0x37, 0x63, 0x37, 0x33, 0x31, 0xc, 0x30, 0xa, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x3, 0x54, 0x45, 0x45, 0x30, 0x1e, 0x17, 0xd, 0x37, 0x30, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x17, 0xd, 0x34, 0x38, 0x30, 0x31, 0x30, 0x31, 0x30, 0x30, 0x30, 0x30, 0x30, 0x30, 0x5a, 0x30, 0x1f, 0x31, 0x1d, 0x30, 0x1b, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x14, 0x41, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x20, 0x4b, 0x65, 0x79, 0x73, 0x74, 0x6f, 0x72, 0x65, 0x20, 0x4b, 0x65, 0x79, 0x30, 0x59, 0x30, 0x13, 0x6, 0x7, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x2, 0x1, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x3, 0x1, 0x7, 0x3, 0x42, 0x0, 0x4, 0xd7, 0x56, 0x2d, 0xfe, 0x9f, 0xea, 0xc1, 0xb2, 0xb3, 0xf7, 0x3, 0x83, 0xe3, 0xf4, 0xff, 0x1e, 0xe8, 0xf3, 0x61, 0xdd, 0xe3, 0x5d, 0xb, 0x7f, 0xf, 0x68, 0x58, 0x82, 0x67, 0x23, 0x16, 0x75, 0x45, 0x40, 0xe8, 0xf6, 0xc4, 0xeb, 0xdf, 0x39, 0x3d, 0x25, 0x3, 0xd, 0xc9, 0xe4, 0xdc, 0xa9, 0xe7, 0xa7, 0x27, 0x3a, 0xe8, 0x9f, 0xc7, 0x14, 0xeb, 0xbf, 0x81, 0xeb, 0xaa, 0x89, 0x6b, 0x40, 0xa3, 0x82, 0x1, 0x83, 0x30, 0x82, 0x1, 0x7f, 0x30, 0xe, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x1, 0x1, 0xff, 0x4, 0x4, 0x3, 0x2, 0x7, 0x80, 0x30, 0x82, 0x1, 0x6b, 0x6, 0xa, 0x2b, 0x6, 0x1, 0x4, 0x1, 0xd6, 0x79, 0x2, 0x1, 0x11, 0x4, 0x82, 0x1, 0x5b, 0x30, 0x82, 0x1, 0x57, 0x2, 0x2, 0x1, 0x2c, 0xa, 0x1, 0x1, 0x2, 0x2, 0x1, 0x2c, 0xa, 0x1, 0x1, 0x4, 0x20, 0x56, 0x52, 0xe2, 0xdc, 0x45, 0x54, 0x9a, 0x96, 0xf9, 0x6a, 0xfa, 0x22, 0x55, 0x2, 0xf8, 0x7f, 0xad, 0xc0, 0x8a, 0x60, 0xbc, 0x2, 0x13, 0x92, 0xc0, 0xbe, 0x8c, 0x50, 0x62, 0xfd, 0x5f, 0x5e, 0x4, 0x0, 0x30, 0x77, 0xbf, 0x85, 0x3d, 0x8, 0x2, 0x6, 0x1, 0x94, 0x70, 0x77, 0x38, 0xa2, 0xbf, 0x85, 0x45, 0x67, 0x4, 0x65, 0x30, 0x63, 0x31, 0x3d, 0x30, 0x1b, 0x4, 0x16, 0x63, 0x6f, 0x6d, 0x2e, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x2e, 0x61, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x2e, 0x67, 0x73, 0x66, 0x2, 0x1, 0x23, 0x30, 0x1e, 0x4, 0x16, 0x63, 0x6f, 0x6d, 0x2e, 0x67, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x2e, 0x61, 0x6e, 0x64, 0x72, 0x6f, 0x69, 0x64, 0x2e, 0x67, 0x6d, 0x73, 0x2, 0x4, 0xe, 0xea, 0x3c, 0xe3, 0x31, 0x22, 0x4, 0x20, 0xf0, 0xfd, 0x6c, 0x5b, 0x41, 0xf, 0x25, 0xcb, 0x25, 0xc3, 0xb5, 0x33, 0x46, 0xc8, 0x97, 0x2f, 0xae, 0x30, 0xf8, 0xee, 0x74, 0x11, 0xdf, 0x91, 0x4, 0x80, 0xad, 0x6b, 0x2d, 0x60, 0xdb, 0x83, 0x30, 0x81, 0xa9, 0xa1, 0x5, 0x31, 0x3, 0x2, 0x1, 0x2, 0xa2, 0x3, 0x2, 0x1, 0x3, 0xa3, 0x4, 0x2, 0x2, 0x1, 0x0, 0xa5, 0x5, 0x31, 0x3, 0x2, 0x1, 0x4, 0xaa, 0x3, 0x2, 0x1, 0x1, 0xbf, 0x83, 0x78, 0x3, 0x2, 0x1, 0x3, 0xbf, 0x83, 0x79, 0x3, 0x2, 0x1, 0xa, 0xbf, 0x85, 0x3e, 0x3, 0x2, 0x1, 0x0, 0xbf, 0x85, 0x40, 0x4c, 0x30, 0x4a, 0x4, 0x20, 0x9d, 0xe2, 0x5f, 0xb0, 0x2b, 0xb5, 0x53, 0xd, 0x44, 0x14, 0x9d, 0x14, 0x84, 0x37, 0xc8, 0x2e, 0x26, 0x7e, 0x55, 0x73, 0x22, 0x53, 0xa, 0xa6, 0xf0, 0x3b, 0xa, 0xc2, 0xe9, 0x29, 0x31, 0xda, 0x1, 0x1, 0xff, 0xa, 0x1, 0x0, 0x4, 0x20, 0xeb, 0x2d, 0x29, 0xc7, 0x46, 0x57, 0x73, 0x9b, 0xf6, 0x6e, 0xc5, 0x5b, 0xe3, 0x9c, 0x3e, 0xe8, 0x88, 0x8c, 0x6d, 0x7c, 0xe9, 0xde, 0xc, 0x87, 0x21, 0x62, 0x92, 0xd6, 0x66, 0xf3, 0xea, 0xb, 0xbf, 0x85, 0x41, 0x5, 0x2, 0x3, 0x2, 0x49, 0xf0, 0xbf, 0x85, 0x42, 0x5, 0x2, 0x3, 0x3, 0x17, 0x5, 0xbf, 0x85, 0x4e, 0x6, 0x2, 0x4, 0x1, 0x34, 0xfd, 0xf9, 0xbf, 0x85, 0x4f, 0x6, 0x2, 0x4, 0x1, 0x34, 0xfd, 0xf9, 0x30, 0xa, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x4, 0x3, 0x2, 0x3, 0x48, 0x0, 0x30, 0x45, 0x2, 0x21, 0x0, 0xcc, 0xdc, 0xfa, 0xc3, 0x24, 0xe8, 0xe2, 0xde, 0x62, 0xc6, 0x8f, 0x46, 0xe3, 0x35, 0xcf, 0xc2, 0x1e, 0x33, 0x48, 0x1f, 0x45, 0xf3, 0x9e, 0x2c, 0x53, 0xf2, 0x83, 0x19, 0x37, 0xf4, 0x77, 0x16, 0x2, 0x20, 0x26, 0x90, 0x55, 0x25, 0xb3, 0x5a, 0x4, 0xd9, 0x2d, 0x6, 0x2f, 0xd7, 0x73, 0x17, 0x24, 0x97, 0x9e, 0xf1, 0x2f, 0x3f, 0x96, 0xe4, 0x1, 0xdd, 0x6e, 0x65, 0x3c, 0xfc, 0xcf, 0xd1, 0xec, 0x6c}, []byte{0x30, 0x82, 0x1, 0xdf, 0x30, 0x82, 0x1, 0x85, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0x11, 0x0, 0xd6, 0x2, 0xa0, 0x3a, 0x67, 0x2d, 0x86, 0x5b, 0xa5, 0xa4, 0x85, 0xe3, 0x3a, 0x20, 0x7c, 0x73, 0x30, 0xa, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x4, 0x3, 0x2, 0x30, 0x29, 0x31, 0x13, 0x30, 0x11, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0xa, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20, 0x4c, 0x4c, 0x43, 0x31, 0x12, 0x30, 0x10, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x9, 0x44, 0x72, 0x6f, 0x69, 0x64, 0x20, 0x43, 0x41, 0x33, 0x30, 0x1e, 0x17, 0xd, 0x32, 0x35, 0x30, 0x31, 0x30, 0x37, 0x31, 0x37, 0x30, 0x38, 0x34, 0x33, 0x5a, 0x17, 0xd, 0x32, 0x35, 0x30, 0x32, 0x30, 0x32, 0x31, 0x30, 0x33, 0x35, 0x32, 0x37, 0x5a, 0x30, 0x39, 0x31, 0x29, 0x30, 0x27, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x20, 0x64, 0x36, 0x30, 0x32, 0x61, 0x30, 0x33, 0x61, 0x36, 0x37, 0x32, 0x64, 0x38, 0x36, 0x35, 0x62, 0x61, 0x35, 0x61, 0x34, 0x38, 0x35, 0x65, 0x33, 0x33, 0x61, 0x32, 0x30, 0x37, 0x63, 0x37, 0x33, 0x31, 0xc, 0x30, 0xa, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0x3, 0x54, 0x45, 0x45, 0x30, 0x59, 0x30, 0x13, 0x6, 0x7, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x2, 0x1, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x3, 0x1, 0x7, 0x3, 0x42, 0x0, 0x4, 0x53, 0xdb, 0x3d, 0x8a, 0xa6, 0xf7, 0x5a, 0xd8, 0xbd, 0x95, 0x42, 0x5, 0xd1, 0x5a, 0x1d, 0x13, 0x20, 0xd2, 0xdc, 0x3b, 0x53, 0x4e, 0xc9, 0x3, 0x51, 0x1c, 0xf, 0xd6, 0x4f, 0xe1, 0xdd, 0x25, 0x2b, 0x63, 0x45, 0x33, 0x80, 0x34, 0xc0, 0x62, 0x92, 0xe2, 0x36, 0xd3, 0x47, 0xf6, 0xee, 0xec, 0xd4, 0xed, 0xfd, 0x8c, 0x85, 0xdf, 0xcf, 0x12, 0xd0, 0xf5, 0x80, 0x25, 0x39, 0x6e, 0xaa, 0x73, 0xa3, 0x7e, 0x30, 0x7c, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x1d, 0xe, 0x4, 0x16, 0x4, 0x14, 0x97, 0xca, 0xbb, 0x19, 0xfc, 0xd, 0x4d, 0x9c, 0x91, 0x91, 0xab, 0x4b, 0x26, 0xd8, 0x6f, 0x67, 0x9b, 0xe3, 0x42, 0xe7, 0x30, 0x1f, 0x6, 0x3, 0x55, 0x1d, 0x23, 0x4, 0x18, 0x30, 0x16, 0x80, 0x14, 0xc5, 0x51, 0x7e, 0x5, 0xd8, 0xcb, 0x63, 0x5b, 0x29, 0x18, 0xac, 0x2, 0x1a, 0xf2, 0xc8, 0x7d, 0x6c, 0x12, 0xfe, 0x92, 0x30, 0xf, 0x6, 0x3, 0x55, 0x1d, 0x13, 0x1, 0x1, 0xff, 0x4, 0x5, 0x30, 0x3, 0x1, 0x1, 0xff, 0x30, 0xe, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x1, 0x1, 0xff, 0x4, 0x4, 0x3, 0x2, 0x2, 0x4, 0x30, 0x19, 0x6, 0xa, 0x2b, 0x6, 0x1, 0x4, 0x1, 0xd6, 0x79, 0x2, 0x1, 0x1e, 0x4, 0xb, 0xa2, 0x1, 0x8, 0x3, 0x66, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x30, 0xa, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x4, 0x3, 0x2, 0x3, 0x48, 0x0, 0x30, 0x45, 0x2, 0x21, 0x0, 0xca, 0xc7, 0x7a, 0x24, 0x3a, 0x8, 0xf1, 0x7e, 0xd, 0x11, 0xda, 0xd1, 0xc1, 0x4c, 0x2d, 0x20, 0xc, 0xbe, 0x84, 0xbc, 0x52, 0x10, 0xa5, 0x15, 0x4b, 0x65, 0xd5, 0x59, 0x2d, 0x82, 0x3f, 0x4e, 0x2, 0x20, 0x53, 0x50, 0x43, 0x33, 0x38, 0x4e, 0x40, 0x3f, 0xc5, 0xa6, 0x63, 0xf0, 0xac, 0xa2, 0x4b, 0xad, 0x67, 0x6f, 0xe6, 0x8e, 0x4, 0xfd, 0xf9, 0xb6, 0x50, 0x37, 0xe0, 0x90, 0x8e, 0x96, 0xff, 0x67}, []byte{0x30, 0x82, 0x1, 0xd8, 0x30, 0x82, 0x1, 0x5d, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0x14, 0x0, 0x85, 0xa, 0xf6, 0xfa, 0xce, 0xe6, 0x22, 0x4, 0x6d, 0xc, 0x74, 0x8b, 0x37, 0x70, 0xaa, 0x55, 0xb0, 0xb6, 0x4d, 0x30, 0xa, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x4, 0x3, 0x3, 0x30, 0x29, 0x31, 0x13, 0x30, 0x11, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0xa, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20, 0x4c, 0x4c, 0x43, 0x31, 0x12, 0x30, 0x10, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x9, 0x44, 0x72, 0x6f, 0x69, 0x64, 0x20, 0x43, 0x41, 0x32, 0x30, 0x1e, 0x17, 0xd, 0x32, 0x34, 0x31, 0x32, 0x30, 0x39, 0x30, 0x36, 0x32, 0x38, 0x35, 0x33, 0x5a, 0x17, 0xd, 0x32, 0x35, 0x30, 0x32, 0x31, 0x37, 0x30, 0x36, 0x32, 0x38, 0x35, 0x32, 0x5a, 0x30, 0x29, 0x31, 0x13, 0x30, 0x11, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0xa, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20, 0x4c, 0x4c, 0x43, 0x31, 0x12, 0x30, 0x10, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x9, 0x44, 0x72, 0x6f, 0x69, 0x64, 0x20, 0x43, 0x41, 0x33, 0x30, 0x59, 0x30, 0x13, 0x6, 0x7, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x2, 0x1, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x3, 0x1, 0x7, 0x3, 0x42, 0x0, 0x4, 0x3e, 0x36, 0xeb, 0xfb, 0x2b, 0x7d, 0xc6, 0x18, 0x1c, 0x6c, 0xa2, 0xd7, 0xa0, 0xdd, 0xd1, 0x2b, 0xed, 0x45, 0x70, 0x28, 0xf0, 0x22, 0x97, 0x8c, 0x3c, 0x96, 0x63, 0x6b, 0x25, 0xb4, 0x76, 0xa3, 0x6d, 0x15, 0xf9, 0x47, 0xa7, 0x6b, 0x26, 0x3b, 0x40, 0xf1, 0x37, 0xfe, 0xb1, 0xec, 0xce, 0xcd, 0x0, 0x66, 0xdc, 0xb, 0xd9, 0x4c, 0xd7, 0xf1, 0x3e, 0x37, 0xac, 0xf9, 0xd4, 0xe0, 0xef, 0xfb, 0xa3, 0x63, 0x30, 0x61, 0x30, 0xe, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x1, 0x1, 0xff, 0x4, 0x4, 0x3, 0x2, 0x2, 0x4, 0x30, 0xf, 0x6, 0x3, 0x55, 0x1d, 0x13, 0x1, 0x1, 0xff, 0x4, 0x5, 0x30, 0x3, 0x1, 0x1, 0xff, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x1d, 0xe, 0x4, 0x16, 0x4, 0x14, 0xc5, 0x51, 0x7e, 0x5, 0xd8, 0xcb, 0x63, 0x5b, 0x29, 0x18, 0xac, 0x2, 0x1a, 0xf2, 0xc8, 0x7d, 0x6c, 0x12, 0xfe, 0x92, 0x30, 0x1f, 0x6, 0x3, 0x55, 0x1d, 0x23, 0x4, 0x18, 0x30, 0x16, 0x80, 0x14, 0xa6, 0xb, 0x86, 0xa4, 0xf0, 0xc8, 0x7f, 0x33, 0xb5, 0x61, 0x39, 0x62, 0xcd, 0x3d, 0xe7, 0xa7, 0x3a, 0x42, 0x83, 0x48, 0x30, 0xa, 0x6, 0x8, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x4, 0x3, 0x3, 0x3, 0x69, 0x0, 0x30, 0x66, 0x2, 0x31, 0x0, 0xae, 0xcd, 0x4d, 0x30, 0x57, 0xf3, 0x64, 0x39, 0xc, 0xe0, 0xab, 0x3c, 0xaa, 0x3f, 0xae, 0x92, 0x24, 0xde, 0xa5, 0x8a, 0x5a, 0xd4, 0xf8, 0xd0, 0xd5, 0x53, 0xaa, 0x8d, 0xa, 0x76, 0xc5, 0xb1, 0x4a, 0x19, 0x40, 0xe9, 0xb8, 0xc0, 0x1c, 0x14, 0xc3, 0xb2, 0xea, 0x7e, 0x85, 0x21, 0xc7, 0xb6, 0x2, 0x31, 0x0, 0xc5, 0x12, 0xfa, 0x87, 0x7d, 0xcd, 0xa1, 0xc0, 0xdd, 0x89, 0xe0, 0x8f, 0x59, 0xe2, 0xb4, 0xc7, 0x88, 0xb9, 0x6c, 0xcb, 0xa8, 0x0, 0xdf, 0x75, 0xaf, 0x10, 0x2c, 0xc8, 0x66, 0x65, 0x4c, 0xa4, 0xa7, 0x32, 0x12, 0x21, 0xe5, 0x9b, 0x87, 0x3d, 0x5d, 0x1d, 0x27, 0x56, 0x40, 0x23, 0x43, 0xfe}, []byte{0x30, 0x82, 0x3, 0x80, 0x30, 0x82, 0x1, 0x68, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0xa, 0x3, 0x88, 0x26, 0x67, 0x60, 0x65, 0x89, 0x96, 0x86, 0xe, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x30, 0x1b, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0x5, 0x13, 0x10, 0x66, 0x39, 0x32, 0x30, 0x30, 0x39, 0x65, 0x38, 0x35, 0x33, 0x62, 0x36, 0x62, 0x30, 0x34, 0x35, 0x30, 0x1e, 0x17, 0xd, 0x32, 0x32, 0x30, 0x31, 0x32, 0x36, 0x32, 0x32, 0x34, 0x39, 0x34, 0x35, 0x5a, 0x17, 0xd, 0x33, 0x37, 0x30, 0x31, 0x32, 0x32, 0x32, 0x32, 0x34, 0x39, 0x34, 0x35, 0x5a, 0x30, 0x29, 0x31, 0x13, 0x30, 0x11, 0x6, 0x3, 0x55, 0x4, 0xa, 0x13, 0xa, 0x47, 0x6f, 0x6f, 0x67, 0x6c, 0x65, 0x20, 0x4c, 0x4c, 0x43, 0x31, 0x12, 0x30, 0x10, 0x6, 0x3, 0x55, 0x4, 0x3, 0x13, 0x9, 0x44, 0x72, 0x6f, 0x69, 0x64, 0x20, 0x43, 0x41, 0x32, 0x30, 0x76, 0x30, 0x10, 0x6, 0x7, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x2, 0x1, 0x6, 0x5, 0x2b, 0x81, 0x4, 0x0, 0x22, 0x3, 0x62, 0x0, 0x4, 0xfb, 0xd9, 0x9b, 0x58, 0x27, 0x42, 0x36, 0x7a, 0xd1, 0x61, 0xa2, 0xeb, 0xed, 0x56, 0xcd, 0x2c, 0x7, 0x3b, 0xd8, 0xb3, 0x1f, 0x2d, 0x84, 0xd0, 0x99, 0x57, 0x5c, 0x2c, 0xc, 0xfb, 0xec, 0x30, 0x34, 0x93, 0x3e, 0x28, 0xfe, 0x33, 0x5c, 0x1, 0x14, 0x3d, 0x34, 0x39, 0x49, 0xf6, 0x6d, 0xf5, 0x39, 0x6f, 0xb2, 0xb6, 0xfe, 0xdd, 0x8d, 0x91, 0x77, 0x36, 0xfb, 0xf5, 0x7d, 0x83, 0xef, 0xa6, 0x75, 0xbe, 0x92, 0xa1, 0x11, 0xa6, 0x27, 0xf0, 0x6, 0xf5, 0x27, 0x18, 0xeb, 0x42, 0xca, 0x48, 0xe, 0xe2, 0x3d, 0x3e, 0xe0, 0x67, 0xfe, 0x7e, 0xab, 0x4b, 0x67, 0x10, 0xf3, 0xd9, 0x24, 0x2a, 0xa3, 0x66, 0x30, 0x64, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x1d, 0xe, 0x4, 0x16, 0x4, 0x14, 0xa6, 0xb, 0x86, 0xa4, 0xf0, 0xc8, 0x7f, 0x33, 0xb5, 0x61, 0x39, 0x62, 0xcd, 0x3d, 0xe7, 0xa7, 0x3a, 0x42, 0x83, 0x48, 0x30, 0x1f, 0x6, 0x3, 0x55, 0x1d, 0x23, 0x4, 0x18, 0x30, 0x16, 0x80, 0x14, 0x36, 0x61, 0xe1, 0x0, 0x7c, 0x88, 0x5, 0x9, 0x51, 0x8b, 0x44, 0x6c, 0x47, 0xff, 0x1a, 0x4c, 0xc9, 0xea, 0x4f, 0x12, 0x30, 0x12, 0x6, 0x3, 0x55, 0x1d, 0x13, 0x1, 0x1, 0xff, 0x4, 0x8, 0x30, 0x6, 0x1, 0x1, 0xff, 0x2, 0x1, 0x2, 0x30, 0xe, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x1, 0x1, 0xff, 0x4, 0x4, 0x3, 0x2, 0x1, 0x6, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x3, 0x82, 0x2, 0x1, 0x0, 0xae, 0x90, 0x76, 0x78, 0xb6, 0xca, 0x1c, 0xd7, 0x12, 0xe9, 0xd, 0xd3, 0x77, 0x73, 0x7b, 0x64, 0x28, 0x15, 0x2c, 0xdd, 0x2a, 0x3, 0xb0, 0x9b, 0xa0, 0xcf, 0x85, 0xb6, 0xee, 0x18, 0x60, 0x7, 0x7, 0xc6, 0x1f, 0x9e, 0xfe, 0xc9, 0x3b, 0x4f, 0xb5, 0xce, 0x83, 0x94, 0xb8, 0x77, 0xd6, 0xbf, 0xc, 0x3e, 0xe6, 0x21, 0x78, 0x4, 0x35, 0x20, 0x4f, 0x7, 0xf7, 0xd5, 0xa8, 0x29, 0xc1, 0x69, 0x20, 0xca, 0x34, 0x9d, 0xe7, 0x52, 0xe0, 0xb5, 0xff, 0xd8, 0x1c, 0x4f, 0xde, 0x69, 0xe2, 0x42, 0x89, 0xa3, 0x4b, 0x6a, 0x72, 0xc5, 0xb0, 0xf1, 0x68, 0xa, 0x54, 0x98, 0x4d, 0xb9, 0x90, 0x8d, 0xed, 0xdf, 0xf3, 0x88, 0x72, 0xe2, 0x45, 0x17, 0x2b, 0x2a, 0xd6, 0xa3, 0xf7, 0x8e, 0x13, 0x1f, 0x3b, 0x32, 0xee, 0xf0, 0x7e, 0xa5, 0x26, 0xa2, 0xee, 0x4, 0xda, 0x87, 0xf4, 0x49, 0xac, 0x34, 0x8, 0xf7, 0x56, 0xc3, 0x50, 0x98, 0x25, 0x77, 0xcf, 0xe9, 0xfd, 0x0, 0x1b, 0x56, 0x76, 0x0, 0xf2, 0x8f, 0xa6, 0xb0, 0xed, 0x2c, 0x44, 0xac, 0x60, 0x17, 0x55, 0x79, 0x42, 0xe0, 0x70, 0x5, 0x37, 0xe9, 0x76, 0x4a, 0xac, 0x86, 0xa2, 0xf7, 0x9f, 0x0, 0x66, 0xd2, 0xca, 0x1f, 0xa8, 0x8f, 0xeb, 0x33, 0xe2, 0x4e, 0x41, 0xab, 0xda, 0xdf, 0xe2, 0x2e, 0xac, 0x7b, 0xdf, 0x93, 0x38, 0xa8, 0x88, 0xe2, 0x74, 0xb8, 0x52, 0x67, 0x4, 0x65, 0x63, 0xa5, 0x91, 0x70, 0x1b, 0xb, 0x26, 0xc8, 0xac, 0x33, 0x97, 0xed, 0xea, 0x44, 0xa9, 0x64, 0x1e, 0x8, 0x0, 0x38, 0xab, 0x77, 0x95, 0x4, 0x38, 0x1a, 0x22, 0x67, 0x38, 0x29, 0xb5, 0xd0, 0x9a, 0xf1, 0x3d, 0xad, 0x99, 0x6b, 0x42, 0xba, 0xfe, 0x4c, 0xa4, 0xe9, 0x8c, 0xaf, 0xbd, 0x17, 0x5, 0x35, 0xa8, 0xb9, 0x7a, 0x3d, 0xe7, 0x30, 0xa2, 0x4f, 0x83, 0xe7, 0x9e, 0x1b, 0x69, 0xd2, 0x4f, 0xa3, 0x7a, 0x78, 0xf1, 0x6e, 0x69, 0xea, 0xd, 0x42, 0x70, 0xf8, 0x5e, 0x46, 0xa0, 0x18, 0x51, 0x10, 0x63, 0x37, 0x2a, 0xbe, 0xc7, 0x45, 0xb2, 0x6b, 0x1b, 0x50, 0x53, 0x20, 0x84, 0x59, 0x7b, 0xcc, 0x45, 0x58, 0x76, 0xc7, 0x2e, 0x4a, 0x0, 0x86, 0xf7, 0x7f, 0x70, 0xc5, 0x6d, 0x49, 0x80, 0xac, 0x86, 0xe5, 0xc7, 0xbb, 0x8, 0x8e, 0x34, 0xdf, 0x6f, 0x30, 0xf4, 0x8b, 0x8e, 0x7f, 0xc9, 0x94, 0x4b, 0xf5, 0xb7, 0x8b, 0x88, 0xd2, 0x7f, 0x1a, 0x82, 0x25, 0xb7, 0xa0, 0xca, 0x8f, 0xd2, 0x28, 0x3a, 0x43, 0x20, 0xb9, 0xe1, 0x25, 0x40, 0x72, 0xd1, 0x3d, 0xdf, 0x37, 0xc9, 0xaf, 0xf4, 0xc, 0x8e, 0xd2, 0xea, 0x9, 0x6, 0x5b, 0xe7, 0xa2, 0x2, 0x15, 0xbf, 0x64, 0x9d, 0xa5, 0x85, 0x2d, 0xf7, 0x6b, 0xf2, 0x3a, 0x21, 0x8c, 0xcb, 0x72, 0x71, 0xfd, 0x94, 0x56, 0x8c, 0x84, 0x5a, 0xee, 0x22, 0x15, 0xdd, 0x59, 0xe3, 0x2c, 0x8f, 0xff, 0xa6, 0x0, 0xab, 0x5d, 0x2, 0xcc, 0x8a, 0x76, 0x56, 0x11, 0x6b, 0x31, 0x57, 0x18, 0x36, 0xe6, 0x7a, 0xe4, 0xf2, 0xed, 0xf6, 0x9d, 0x97, 0x7a, 0xec, 0x32, 0x4d, 0xde, 0x4d, 0x85, 0xb9, 0xf4, 0x5b, 0xb2, 0x61, 0xd9, 0x97, 0x2f, 0x42, 0x2e, 0x8d, 0xf, 0xef, 0xd9, 0x64, 0x20, 0x40, 0xb7, 0xae, 0x27, 0xcb, 0xaa, 0xa2, 0x5e, 0xf, 0x21, 0xd0, 0x35, 0xa0, 0x4c, 0x50, 0x2e, 0x81, 0x48, 0x36, 0x79, 0x44, 0x90, 0xad, 0x32, 0x34, 0x99, 0xc3, 0x6c, 0x4b, 0xe7, 0x3d, 0x17, 0xf7, 0xfc, 0xc1, 0xb0, 0xa5, 0x4f, 0x90, 0xc5, 0xba, 0xd6, 0x13, 0x21, 0x2b, 0x35, 0x66, 0x60, 0xd2, 0xde, 0xd4, 0x32, 0x50, 0x0, 0xd4, 0x4f, 0xf9, 0xe9, 0xeb, 0xbb, 0x98, 0xb5, 0xa1, 0x79}, []byte{0x30, 0x82, 0x5, 0x1c, 0x30, 0x82, 0x3, 0x4, 0xa0, 0x3, 0x2, 0x1, 0x2, 0x2, 0x9, 0x0, 0xd5, 0xf, 0xf2, 0x5b, 0xa3, 0xf2, 0xd6, 0xb3, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x30, 0x1b, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0x5, 0x13, 0x10, 0x66, 0x39, 0x32, 0x30, 0x30, 0x39, 0x65, 0x38, 0x35, 0x33, 0x62, 0x36, 0x62, 0x30, 0x34, 0x35, 0x30, 0x1e, 0x17, 0xd, 0x31, 0x39, 0x31, 0x31, 0x32, 0x32, 0x32, 0x30, 0x33, 0x37, 0x35, 0x38, 0x5a, 0x17, 0xd, 0x33, 0x34, 0x31, 0x31, 0x31, 0x38, 0x32, 0x30, 0x33, 0x37, 0x35, 0x38, 0x5a, 0x30, 0x1b, 0x31, 0x19, 0x30, 0x17, 0x6, 0x3, 0x55, 0x4, 0x5, 0x13, 0x10, 0x66, 0x39, 0x32, 0x30, 0x30, 0x39, 0x65, 0x38, 0x35, 0x33, 0x62, 0x36, 0x62, 0x30, 0x34, 0x35, 0x30, 0x82, 0x2, 0x22, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0x1, 0x5, 0x0, 0x3, 0x82, 0x2, 0xf, 0x0, 0x30, 0x82, 0x2, 0xa, 0x2, 0x82, 0x2, 0x1, 0x0, 0xaf, 0xb6, 0xc7, 0x82, 0x2b, 0xb1, 0xa7, 0x1, 0xec, 0x2b, 0xb4, 0x2e, 0x8b, 0xcc, 0x54, 0x16, 0x63, 0xab, 0xef, 0x98, 0x2f, 0x32, 0xc7, 0x7f, 0x75, 0x31, 0x3, 0xc, 0x97, 0x52, 0x4b, 0x1b, 0x5f, 0xe8, 0x9, 0xfb, 0xc7, 0x2a, 0xa9, 0x45, 0x1f, 0x74, 0x3c, 0xbd, 0x9a, 0x6f, 0x13, 0x35, 0x74, 0x4a, 0xa5, 0x5e, 0x77, 0xf6, 0xb6, 0xac, 0x35, 0x35, 0xee, 0x17, 0xc2, 0x5e, 0x63, 0x95, 0x17, 0xdd, 0x9c, 0x92, 0xe6, 0x37, 0x4a, 0x53, 0xcb, 0xfe, 0x25, 0x8f, 0x8f, 0xfb, 0xb6, 0xfd, 0x12, 0x93, 0x78, 0xa2, 0x2a, 0x4c, 0xa9, 0x9c, 0x45, 0x2d, 0x47, 0xa5, 0x9f, 0x32, 0x1, 0xf4, 0x41, 0x97, 0xca, 0x1c, 0xcd, 0x7e, 0x76, 0x2f, 0xb2, 0xf5, 0x31, 0x51, 0xb6, 0xfe, 0xb2, 0xff, 0xfd, 0x2b, 0x6f, 0xe4, 0xfe, 0x5b, 0xc6, 0xbd, 0x9e, 0xc3, 0x4b, 0xfe, 0x8, 0x23, 0x9d, 0xaa, 0xfc, 0xeb, 0x8e, 0xb5, 0xa8, 0xed, 0x2b, 0x3a, 0xcd, 0x9c, 0x5e, 0x3a, 0x77, 0x90, 0xe1, 0xb5, 0x14, 0x42, 0x79, 0x31, 0x59, 0x85, 0x98, 0x11, 0xad, 0x9e, 0xb2, 0xa9, 0x6b, 0xbd, 0xd7, 0xa5, 0x7c, 0x93, 0xa9, 0x1c, 0x41, 0xfc, 0xcd, 0x27, 0xd6, 0x7f, 0xd6, 0xf6, 0x71, 0xaa, 0xb, 0x81, 0x52, 0x61, 0xad, 0x38, 0x4f, 0xa3, 0x79, 0x44, 0x86, 0x46, 0x4, 0xdd, 0xb3, 0xd8, 0xc4, 0xf9, 0x20, 0xa1, 0x9b, 0x16, 0x56, 0xc2, 0xf1, 0x4a, 0xd6, 0xd0, 0x3c, 0x56, 0xec, 0x6, 0x8, 0x99, 0x4, 0x1c, 0x1e, 0xd1, 0xa5, 0xfe, 0x6d, 0x34, 0x40, 0xb5, 0x56, 0xba, 0xd1, 0xd0, 0xa1, 0x52, 0x58, 0x9c, 0x53, 0xe5, 0x5d, 0x37, 0x7, 0x62, 0xf0, 0x12, 0x2e, 0xef, 0x91, 0x86, 0x1b, 0x1b, 0xe, 0x6c, 0x4c, 0x80, 0x92, 0x74, 0x99, 0xc0, 0xe9, 0xbe, 0xc0, 0xb8, 0x3e, 0x3b, 0xc1, 0xf9, 0x3c, 0x72, 0xc0, 0x49, 0x60, 0x4b, 0xbd, 0x2f, 0x13, 0x45, 0xe6, 0x2c, 0x3f, 0x8e, 0x26, 0xdb, 0xec, 0x6, 0xc9, 0x47, 0x66, 0xf3, 0xc1, 0x28, 0x23, 0x9d, 0x4f, 0x43, 0x12, 0xfa, 0xd8, 0x12, 0x38, 0x87, 0xe0, 0x6b, 0xec, 0xf5, 0x67, 0x58, 0x3b, 0xf8, 0x35, 0x5a, 0x81, 0xfe, 0xea, 0xba, 0xf9, 0x9a, 0x83, 0xc8, 0xdf, 0x3e, 0x2a, 0x32, 0x2a, 0xfc, 0x67, 0x2b, 0xf1, 0x20, 0xb1, 0x35, 0x15, 0x8b, 0x68, 0x21, 0xce, 0xaf, 0x30, 0x9b, 0x6e, 0xee, 0x77, 0xf9, 0x88, 0x33, 0xb0, 0x18, 0xda, 0xa1, 0xe, 0x45, 0x1f, 0x6, 0xa3, 0x74, 0xd5, 0x7, 0x81, 0xf3, 0x59, 0x8, 0x29, 0x66, 0xbb, 0x77, 0x8b, 0x93, 0x8, 0x94, 0x26, 0x98, 0xe7, 0x4e, 0xb, 0xcd, 0x24, 0x62, 0x8a, 0x1, 0xc2, 0xcc, 0x3, 0xe5, 0x1f, 0xb, 0x3e, 0x5b, 0x4a, 0xc1, 0xe4, 0xdf, 0x9e, 0xaf, 0x9f, 0xf6, 0xa4, 0x92, 0xa7, 0x7c, 0x14, 0x83, 0x88, 0x28, 0x85, 0x1, 0x5b, 0x42, 0x2c, 0xe6, 0x7b, 0x80, 0xb8, 0x8c, 0x9b, 0x48, 0xe1, 0x3b, 0x60, 0x7a, 0xb5, 0x45, 0xc7, 0x23, 0xff, 0x8c, 0x44, 0xf8, 0xf2, 0xd3, 0x68, 0xb9, 0xf6, 0x52, 0xd, 0x31, 0x14, 0x5e, 0xbf, 0x9e, 0x86, 0x2a, 0xd7, 0x1d, 0xf6, 0xa3, 0xbf, 0xd2, 0x45, 0x9, 0x59, 0xd6, 0x53, 0x74, 0xd, 0x97, 0xa1, 0x2f, 0x36, 0x8b, 0x13, 0xef, 0x66, 0xd5, 0xd0, 0xa5, 0x4a, 0x6e, 0x2f, 0x5d, 0x9a, 0x6f, 0xef, 0x44, 0x68, 0x32, 0xbc, 0x67, 0x84, 0x47, 0x25, 0x86, 0x1f, 0x9, 0x3d, 0xd0, 0xe6, 0xf3, 0x40, 0x5d, 0xa8, 0x96, 0x43, 0xef, 0xf, 0x4d, 0x69, 0xb6, 0x42, 0x0, 0x51, 0xfd, 0xb9, 0x30, 0x49, 0x67, 0x3e, 0x36, 0x95, 0x5, 0x80, 0xd3, 0xcd, 0xf4, 0xfb, 0xd0, 0x8b, 0xc5, 0x84, 0x83, 0x95, 0x26, 0x0, 0x63, 0x2, 0x3, 0x1, 0x0, 0x1, 0xa3, 0x63, 0x30, 0x61, 0x30, 0x1d, 0x6, 0x3, 0x55, 0x1d, 0xe, 0x4, 0x16, 0x4, 0x14, 0x36, 0x61, 0xe1, 0x0, 0x7c, 0x88, 0x5, 0x9, 0x51, 0x8b, 0x44, 0x6c, 0x47, 0xff, 0x1a, 0x4c, 0xc9, 0xea, 0x4f, 0x12, 0x30, 0x1f, 0x6, 0x3, 0x55, 0x1d, 0x23, 0x4, 0x18, 0x30, 0x16, 0x80, 0x14, 0x36, 0x61, 0xe1, 0x0, 0x7c, 0x88, 0x5, 0x9, 0x51, 0x8b, 0x44, 0x6c, 0x47, 0xff, 0x1a, 0x4c, 0xc9, 0xea, 0x4f, 0x12, 0x30, 0xf, 0x6, 0x3, 0x55, 0x1d, 0x13, 0x1, 0x1, 0xff, 0x4, 0x5, 0x30, 0x3, 0x1, 0x1, 0xff, 0x30, 0xe, 0x6, 0x3, 0x55, 0x1d, 0xf, 0x1, 0x1, 0xff, 0x4, 0x4, 0x3, 0x2, 0x2, 0x4, 0x30, 0xd, 0x6, 0x9, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0xd, 0x1, 0x1, 0xb, 0x5, 0x0, 0x3, 0x82, 0x2, 0x1, 0x0, 0x4e, 0x31, 0xa0, 0x5c, 0xf2, 0x8b, 0xa6, 0x5d, 0xbd, 0xaf, 0xa1, 0xce, 0xd7, 0x9, 0x69, 0xee, 0x5c, 0xa8, 0x41, 0x4, 0xad, 0xde, 0xd8, 0xa3, 0x6, 0xcf, 0x7f, 0x6d, 0xee, 0x50, 0x37, 0x5d, 0x74, 0x5e, 0xd9, 0x92, 0xcb, 0x2, 0x42, 0xcc, 0xe7, 0x2d, 0xc9, 0xee, 0xd5, 0x11, 0x91, 0xfe, 0x5a, 0xd5, 0x2b, 0xad, 0x7d, 0xd3, 0xb2, 0x5c, 0x9, 0x9e, 0x13, 0xa4, 0x91, 0xa3, 0xcd, 0xd4, 0x87, 0xa5, 0xac, 0xce, 0x87, 0x66, 0x32, 0x4c, 0x4a, 0xe4, 0x63, 0x38, 0x24, 0x6a, 0xe7, 0xb7, 0x8a, 0x41, 0x8a, 0xcb, 0xb9, 0x8a, 0x5, 0xc4, 0xc9, 0xd6, 0x96, 0xee, 0xaa, 0xb6, 0x9, 0xd0, 0xba, 0xc, 0xe1, 0xa3, 0x1b, 0xe9, 0x84, 0x90, 0xdf, 0x3f, 0x4c, 0xe, 0xa9, 0xdd, 0xc9, 0xe8, 0x2f, 0xfb, 0xf, 0xcb, 0x3e, 0x9e, 0xbd, 0xd8, 0xcb, 0x95, 0x27, 0x89, 0xf2, 0xb1, 0x41, 0x1f, 0xac, 0x56, 0xc8, 0x86, 0x42, 0x6e, 0xb7, 0x29, 0x60, 0x42, 0x73, 0x5d, 0xa5, 0xe, 0x11, 0xac, 0x71, 0x5f, 0x18, 0x18, 0xcf, 0x9f, 0xdc, 0x4e, 0x25, 0x4a, 0x37, 0x63, 0x35, 0x1b, 0x6a, 0x24, 0x40, 0x15, 0x8, 0x61, 0x26, 0x3a, 0x6e, 0x31, 0xb, 0xe1, 0xa5, 0xd, 0xe5, 0xc7, 0xe8, 0xee, 0x88, 0xf, 0xdd, 0x4b, 0xe5, 0x88, 0x4a, 0x37, 0x12, 0x8d, 0x18, 0x83, 0xb, 0xb3, 0x47, 0x6b, 0xf4, 0x29, 0x1e, 0x82, 0xd5, 0xc6, 0x6a, 0x64, 0x94, 0x93, 0x9e, 0x8, 0x48, 0xb, 0xfb, 0xc0, 0xf, 0x7d, 0x8a, 0x74, 0xd4, 0x3e, 0x73, 0x73, 0x7e, 0xbe, 0x5d, 0x8e, 0x4e, 0xc5, 0x15, 0x30, 0x2d, 0x46, 0x89, 0x69, 0x27, 0x80, 0xdc, 0x75, 0x38, 0xed, 0x7e, 0x91, 0x75, 0xbe, 0x61, 0x39, 0xe7, 0x4d, 0x43, 0xad, 0x38, 0x8b, 0x30, 0x50, 0xff, 0xd5, 0xa9, 0xde, 0x52, 0x62, 0x0, 0x8, 0x98, 0xc0, 0x1f, 0x63, 0xc5, 0x3d, 0xfe, 0x22, 0x20, 0x91, 0x8, 0xfa, 0x4f, 0x65, 0xba, 0x16, 0xc4, 0x9c, 0xcb, 0xde, 0x8, 0x37, 0xd7, 0xc5, 0x84, 0x4d, 0x54, 0xb7, 0x39, 0x8b, 0xa0, 0x12, 0x2e, 0x50, 0x5b, 0x15, 0x5c, 0x93, 0x13, 0xcf, 0xe2, 0x6e, 0x72, 0xd8, 0x7e, 0x22, 0xaa, 0x16, 0x16, 0xe6, 0xbd, 0xbf, 0x54, 0x7d, 0xdf, 0xf9, 0x3d, 0xf2, 0x9e, 0x35, 0xa6, 0x3b, 0x45, 0x5f, 0xe1, 0xfc, 0xe, 0xc9, 0x55, 0x81, 0xf3, 0xf4, 0xf7, 0xbb, 0xe3, 0xbb, 0x82, 0x83, 0x96, 0xa3, 0x7a, 0xe3, 0x15, 0x75, 0x82, 0xbc, 0x37, 0x64, 0xb9, 0x78, 0xa, 0x23, 0x9e, 0xfc, 0xf, 0x75, 0xa1, 0xe2, 0xe6, 0xd9, 0x41, 0xce, 0xab, 0xac, 0x27, 0xdd, 0xeb, 0x1, 0xe2, 0xbd, 0x84, 0x21, 0x2, 0x9b, 0xea, 0x34, 0xd5, 0x1a, 0xee, 0x6c, 0x60, 0x27, 0x1d, 0x5a, 0x95, 0xeb, 0xd0, 0x5, 0x15, 0xa9, 0xc0, 0x1, 0x3d, 0xd8, 0xb, 0xf8, 0x7e, 0xea, 0x26, 0xb, 0x81, 0xc3, 0x4f, 0x68, 0x8e, 0x6e, 0xb1, 0x34, 0x8a, 0xf0, 0xd8, 0xea, 0x1c, 0xac, 0x32, 0xac, 0xb9, 0xd9, 0x3f, 0xa2, 0x4a, 0xff, 0x3, 0xa, 0x84, 0xc8, 0xf2, 0xb0, 0xf5, 0x69, 0xcc, 0x95, 0x8, 0xb, 0x20, 0xac, 0x35, 0xac, 0xe0, 0xc6, 0xd8, 0xdb, 0xd4, 0xf6, 0x84, 0x77, 0x19, 0x51, 0x9d, 0x32, 0x45, 0x1, 0x66, 0xeb, 0x4b, 0xf1, 0x5b, 0x85, 0x90, 0x44, 0x50, 0x1a, 0xde, 0xaf, 0x43, 0x63, 0x82, 0xc3, 0x4b, 0x15, 0xe3, 0xb5, 0x4c, 0x92, 0xe6, 0x1b, 0x69, 0xc2, 0xbf, 0xc7, 0x26, 0x45, 0x89, 0x17, 0x2b, 0x3c, 0x93, 0xdb, 0xe3, 0x5c, 0xe0, 0x6d, 0x8, 0xfd, 0x5c, 0x1, 0x32, 0x2c, 0xa0, 0x87, 0x7b, 0x1d, 0x12, 0x74, 0x3a, 0xf1, 0xfa, 0xd5, 0x94, 0xe, 0xa1, 0xbc, 0x2, 0xdd, 0x89, 0x1c}},
			"",
		},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			var (
				attestationType string
				x5cs            []any
				err             error
			)

			if tc.mds {
				var mds *mocks.MockMetadataProvider

				ctrl := gomock.NewController(t)

				mds = mocks.NewMockMetadataProvider(ctrl)

				if tc.setup != nil {
					tc.setup(t, mds)
				}

				attestationType, x5cs, err = attestationFormatValidationHandlerAndroidKey(tc.args.att, tc.args.clientDataHash, mds)
			} else {
				if tc.setup != nil {
					tc.setup(t, nil)
				}

				attestationType, x5cs, err = attestationFormatValidationHandlerAndroidKey(tc.args.att, tc.args.clientDataHash, nil)
			}

			if tc.err != "" {
				assert.EqualError(t, err, tc.err)
			} else {
				assert.NoError(t, err)
			}

			assert.Equal(t, tc.expected, attestationType)
			assert.Equal(t, tc.x5cs, x5cs)
		})
	}
}

var androidKeyTestResponse0 = map[string]string{
	"success": `{
		"rawId": "U5cxFNxLbU9-SAi1K7k9atYwXhghkAMbxpL__VPtBlw",
		"id": "U5cxFNxLbU9-SAi1K7k9atYwXhghkAMbxpL__VPtBlw",
		"response": {
		  "clientDataJSON": "eyJvcmlnaW4iOiJodHRwczovL2xvY2FsaG9zdDo0NDMyOSIsImNoYWxsZW5nZSI6IjlNNWY3bGp5MVl2UWNzOE9pV1FWQ3ciLCJ0eXBlIjoid2ViYXV0aG4uY3JlYXRlIn0",
		  "attestationObject": "o2NmbXRrYW5kcm9pZC1rZXlnYXR0U3RtdKNjYWxnJmNzaWdYSDBGAiEAlbQ-jtl8o9GtEstcEFH1Z_NlYsTYSn96lilEF17oEsMCIQDza5_axjn2jKZO63RlVf47DDFZbceW9b_tsh1nwOYQbmN4NWOCWQMFMIIDATCCAqegAwIBAgIBATAKBggqhkjOPQQDAjCBzjFFMEMGA1UEAww8RkFLRSBBbmRyb2lkIEtleXN0b3JlIFNvZnR3YXJlIEF0dGVzdGF0aW9uIEludGVybWVkaWF0ZSBGQUtFMTEwLwYJKoZIhvcNAQkBFiJjb25mb3JtYW5jZS10b29sc0BmaWRvYWxsaWFuY2Uub3JnMRYwFAYDVQQKDA1GSURPIEFsbGlhbmNlMQwwCgYDVQQLDANDV0cxCzAJBgNVBAYTAlVTMQswCQYDVQQIDAJNWTESMBAGA1UEBwwJV2FrZWZpZWxkMCAXDTcwMDIwMTAwMDAwMFoYDzIwOTkwMTMxMjM1OTU5WjApMScwJQYDVQQDDB5GQUtFIEFuZHJvaWQgS2V5c3RvcmUgS2V5IEZBS0UwWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAAQbh-BQBJz7JeQ27dVvu3tyRieiEeXyDYoaWatRdy_D7q3TK96jumKlwIl5ZA2zHmKNLz4K2zsANq1X4tHp8MNZo4IBFjCCARIwCwYDVR0PBAQDAgeAMIHhBgorBgEEAdZ5AgERBIHSMIHPAgECCgEAAgEBCgEABCDc0UoXtU1CwwItW3ne2faKDcFCabFI31BufXEFVK_ENwQAMGm_hT0IAgYBXtPjz6C_hUVZBFcwVTEvMC0EKGNvbS5hbmRyb2lkLmtleXN0b3JlLmFuZHJvaWRrZXlzdG9yZWRlbW8CAQExIgQgdM_LUHSI9SkQhZHHpQWRnzJ3MvvB2ANSauqYAAbS2JgwMqEFMQMCAQKiAwIBA6MEAgIBAKUFMQMCAQSqAwIBAb-DeAMCAQK_hT4DAgEAv4U_AgUAMB8GA1UdIwQYMBaAFFKaGzLgVqrNUQ_vX4A3BovykSMdMAoGCCqGSM49BAMCA0gAMEUCIQDAPV7eQIWfL5BCmj82NszDlQ2IJsOZq_WxidwxD7On_QIgFipplgUF6OHvmHiDdaHJfFweeo60OtCDGDftjQEmF7FZAu4wggLqMIICkaADAgECAgECMAoGCCqGSM49BAMCMIHGMT0wOwYDVQQDDDRGQUtFIEFuZHJvaWQgS2V5c3RvcmUgU29mdHdhcmUgQXR0ZXN0YXRpb24gUm9vdCBGQUtFMTEwLwYJKoZIhvcNAQkBFiJjb25mb3JtYW5jZS10b29sc0BmaWRvYWxsaWFuY2Uub3JnMRYwFAYDVQQKDA1GSURPIEFsbGlhbmNlMQwwCgYDVQQLDANDV0cxCzAJBgNVBAYTAlVTMQswCQYDVQQIDAJNWTESMBAGA1UEBwwJV2FrZWZpZWxkMB4XDTE4MDUwOTEyMzE0NFoXDTQ1MDkyNDEyMzE0NFowgc4xRTBDBgNVBAMMPEZBS0UgQW5kcm9pZCBLZXlzdG9yZSBTb2Z0d2FyZSBBdHRlc3RhdGlvbiBJbnRlcm1lZGlhdGUgRkFLRTExMC8GCSqGSIb3DQEJARYiY29uZm9ybWFuY2UtdG9vbHNAZmlkb2FsbGlhbmNlLm9yZzEWMBQGA1UECgwNRklETyBBbGxpYW5jZTEMMAoGA1UECwwDQ1dHMQswCQYDVQQGEwJVUzELMAkGA1UECAwCTVkxEjAQBgNVBAcMCVdha2VmaWVsZDBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABKtQYStiTRe7w7UbBEk7BUkLjB-LnbzzebLe3KB8UqHXtg3TIXXcK37dvCbbCNVfhvZxtpTcME2kooqMTgOm9cejZjBkMBIGA1UdEwEB_wQIMAYBAf8CAQAwDgYDVR0PAQH_BAQDAgKEMB0GA1UdDgQWBBSj0qos7w2M8iQC1Ry0YLy_alskFDAfBgNVHSMEGDAWgBRSmhsy4FaqzVEP71-ANwaL8pEjHTAKBggqhkjOPQQDAgNHADBEAiBp3Z6j8YH7Qko5rRoK37nS4zPXhv65RWBV-j3MmXi50gIgPtMPpvcGtVbpFCQqsGbyhxPdkji8ltcYXQVfMhdUpRZoYXV0aERhdGFYpEmWDeWIDoxodDQXD2R2YFuP5K65ooYyx5lc87qDHZdjQQAAAFpVDktUqkdAn5qVGrdsEwExACBTlzEU3EttT35ICLUruT1q1jBeGCGQAxvGkv_9U-0GXKUBAgMmIAEhWCAbh-BQBJz7JeQ27dVvu3tyRieiEeXyDYoaWatRdy_D7iJYIK3TK96jumKlwIl5ZA2zHmKNLz4K2zsANq1X4tHp8MNZ"
		},
		"type": "public-key"
	  }`,
}

var androidKeyTestResponse1 = map[string]string{
	"success": `{
		"id": "V51GE29tGbhby7sbg1cZ_qL8V8njqEsXpAnwQBobvgw",
		"rawId": "V51GE29tGbhby7sbg1cZ_qL8V8njqEsXpAnwQBobvgw",
		"response": {
		  "attestationObject": "o2NmbXRrYW5kcm9pZC1rZXlnYXR0U3RtdKNjYWxnJmNzaWdYRzBFAiAbZhfcF0KSXj5rdEevvnBcC8ZfRQlNl9XYWRTiIGKSHwIhAIerc7jWjOF_lJ71n_GAcaHwDUtPxkjAAdYugnZ4QxkmY3g1Y4JZAxowggMWMIICvaADAgECAgEBMAoGCCqGSM49BAMCMIHkMUUwQwYDVQQDDDxGQUtFIEFuZHJvaWQgS2V5c3RvcmUgU29mdHdhcmUgQXR0ZXN0YXRpb24gSW50ZXJtZWRpYXRlIEZBS0UxMTAvBgkqhkiG9w0BCQEWImNvbmZvcm1hbmNlLXRvb2xzQGZpZG9hbGxpYW5jZS5vcmcxFjAUBgNVBAoMDUZJRE8gQWxsaWFuY2UxIjAgBgNVBAsMGUF1dGhlbnRpY2F0b3IgQXR0ZXN0YXRpb24xCzAJBgNVBAYTAlVTMQswCQYDVQQIDAJNWTESMBAGA1UEBwwJV2FrZWZpZWxkMCAXDTcwMDIwMTAwMDAwMFoYDzIwOTkwMTMxMjM1OTU5WjApMScwJQYDVQQDDB5GQUtFIEFuZHJvaWQgS2V5c3RvcmUgS2V5IEZBS0UwWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAARuowgSu5AoRj8Vi_ZNSFBbGUZJXFG9MkDT6jADlr7tOK9NEgjVX53-ergXpyPaFZrAR9py-xnzfjILn_Kzb8Iqo4IBFjCCARIwCwYDVR0PBAQDAgeAMIHhBgorBgEEAdZ5AgERBIHSMIHPAgECCgEAAgEBCgEABCCfVEl83pSDSerk9I3pcICNTdzc5N3u4jt21cXdzBuJjgQAMGm_hT0IAgYBXtPjz6C_hUVZBFcwVTEvMC0EKGNvbS5hbmRyb2lkLmtleXN0b3JlLmFuZHJvaWRrZXlzdG9yZWRlbW8CAQExIgQgdM_LUHSI9SkQhZHHpQWRnzJ3MvvB2ANSauqYAAbS2JgwMqEFMQMCAQKiAwIBA6MEAgIBAKUFMQMCAQSqAwIBAb-DeAMCAQK_hT4DAgEAv4U_AgUAMB8GA1UdIwQYMBaAFKPSqizvDYzyJALVHLRgvL9qWyQUMAoGCCqGSM49BAMCA0cAMEQCIC7WHb2PyULnjp1M1TVI3Wti_eDhe6sFweuQAdecXtHhAiAS_eZkFsx_VNsrTu3XfZ2D7wIt-vT6nTljfHZ4zqU5xlkDGDCCAxQwggK6oAMCAQICAQIwCgYIKoZIzj0EAwIwgdwxPTA7BgNVBAMMNEZBS0UgQW5kcm9pZCBLZXlzdG9yZSBTb2Z0d2FyZSBBdHRlc3RhdGlvbiBSb290IEZBS0UxMTAvBgkqhkiG9w0BCQEWImNvbmZvcm1hbmNlLXRvb2xzQGZpZG9hbGxpYW5jZS5vcmcxFjAUBgNVBAoMDUZJRE8gQWxsaWFuY2UxIjAgBgNVBAsMGUF1dGhlbnRpY2F0b3IgQXR0ZXN0YXRpb24xCzAJBgNVBAYTAlVTMQswCQYDVQQIDAJNWTESMBAGA1UEBwwJV2FrZWZpZWxkMB4XDTE5MDQyNTA1NDkzMloXDTQ2MDkxMDA1NDkzMlowgeQxRTBDBgNVBAMMPEZBS0UgQW5kcm9pZCBLZXlzdG9yZSBTb2Z0d2FyZSBBdHRlc3RhdGlvbiBJbnRlcm1lZGlhdGUgRkFLRTExMC8GCSqGSIb3DQEJARYiY29uZm9ybWFuY2UtdG9vbHNAZmlkb2FsbGlhbmNlLm9yZzEWMBQGA1UECgwNRklETyBBbGxpYW5jZTEiMCAGA1UECwwZQXV0aGVudGljYXRvciBBdHRlc3RhdGlvbjELMAkGA1UEBhMCVVMxCzAJBgNVBAgMAk1ZMRIwEAYDVQQHDAlXYWtlZmllbGQwWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAASrUGErYk0Xu8O1GwRJOwVJC4wfi52883my3tygfFKh17YN0yF13Ct-3bwm2wjVX4b2cbaU3DBNpKKKjE4DpvXHo2MwYTAPBgNVHRMBAf8EBTADAQH_MA4GA1UdDwEB_wQEAwIChDAdBgNVHQ4EFgQUo9KqLO8NjPIkAtUctGC8v2pbJBQwHwYDVR0jBBgwFoAUUpobMuBWqs1RD-9fgDcGi_KRIx0wCgYIKoZIzj0EAwIDSAAwRQIhALFvLkAvtHrObTmN8P0-yLIT496P_weSEEbB6vCJWSh9AiBu-UOorCeLcF4WixOG9E5Li2nXe4uM2q6mbKGkll8u-WhhdXRoRGF0YVikPdxHEOnAiLIp26idVjIguzn3Ipr_RlsKZWsa-5qK-KBBAAAAYFUOS1SqR0CfmpUat2wTATEAIFedRhNvbRm4W8u7G4NXGf6i_FfJ46hLF6QJ8EAaG74MpQECAyYgASFYIG6jCBK7kChGPxWL9k1IUFsZRklcUb0yQNPqMAOWvu04Ilggr00SCNVfnf56uBenI9oVmsBH2nL7GfN-Mguf8rNvwio",
		  "clientDataJSON": "eyJvcmlnaW4iOiJodHRwczovL2Rldi5kb250bmVlZGEucHciLCJjaGFsbGVuZ2UiOiI0YWI3ZGZkMS1hNjk1LTQ3NzctOTg1Zi1hZDI5OTM4MjhlOTkiLCJ0eXBlIjoid2ViYXV0aG4uY3JlYXRlIn0"
		},
		"type": "public-key"
	  }`,
}

var androidKeyTestResponse2 = map[string]string{
	"success": `{
		"id": "AYNe4CBKc8H30FuAb8uaht6JbEQfbSBnS0SX7B6MFg8ofI92oR5lheRDJCgwY-JqB_QSJtezdhMbf8Wzt_La5N0",
		"rawId": "AYNe4CBKc8H30FuAb8uaht6JbEQfbSBnS0SX7B6MFg8ofI92oR5lheRDJCgwY-JqB_QSJtezdhMbf8Wzt_La5N0",
		"response": {
		  "attestationObject": "o2NmbXRrYW5kcm9pZC1rZXlnYXR0U3RtdKNjYWxnJmNzaWdYSDBGAiEAs9Aufj5f5HyLKEFsgfmqyaXfAih-hGuTJqgmxZGijzYCIQDAMddAq1gwH3MtesYR6WE6IAockRz8ilR7CFw_kgdmv2N4NWOFWQLQMIICzDCCAnKgAwIBAgIBATAKBggqhkjOPQQDAjA5MSkwJwYDVQQDEyBkNjAyYTAzYTY3MmQ4NjViYTVhNDg1ZTMzYTIwN2M3MzEMMAoGA1UEChMDVEVFMB4XDTcwMDEwMTAwMDAwMFoXDTQ4MDEwMTAwMDAwMFowHzEdMBsGA1UEAxMUQW5kcm9pZCBLZXlzdG9yZSBLZXkwWTATBgcqhkjOPQIBBggqhkjOPQMBBwNCAATXVi3-n-rBsrP3A4Pj9P8e6PNh3eNdC38PaFiCZyMWdUVA6PbE6985PSUDDcnk3Knnpyc66J_HFOu_geuqiWtAo4IBgzCCAX8wDgYDVR0PAQH_BAQDAgeAMIIBawYKKwYBBAHWeQIBEQSCAVswggFXAgIBLAoBAQICASwKAQEEIFZS4txFVJqW-Wr6IlUC-H-twIpgvAITksC-jFBi_V9eBAAwd7-FPQgCBgGUcHc4or-FRWcEZTBjMT0wGwQWY29tLmdvb2dsZS5hbmRyb2lkLmdzZgIBIzAeBBZjb20uZ29vZ2xlLmFuZHJvaWQuZ21zAgQO6jzjMSIEIPD9bFtBDyXLJcO1M0bIly-uMPjudBHfkQSArWstYNuDMIGpoQUxAwIBAqIDAgEDowQCAgEApQUxAwIBBKoDAgEBv4N4AwIBA7-DeQMCAQq_hT4DAgEAv4VATDBKBCCd4l-wK7VTDUQUnRSEN8guJn5VcyJTCqbwOwrC6Skx2gEB_woBAAQg6y0px0ZXc5v2bsVb45w-6IiMbXzp3gyHIWKS1mbz6gu_hUEFAgMCSfC_hUIFAgMDFwW_hU4GAgQBNP35v4VPBgIEATT9-TAKBggqhkjOPQQDAgNIADBFAiEAzNz6wyTo4t5ixo9G4zXPwh4zSB9F854sU_KDGTf0dxYCICaQVSWzWgTZLQYv13MXJJee8S8_luQB3W5lPPzP0exsWQHjMIIB3zCCAYWgAwIBAgIRANYCoDpnLYZbpaSF4zogfHMwCgYIKoZIzj0EAwIwKTETMBEGA1UEChMKR29vZ2xlIExMQzESMBAGA1UEAxMJRHJvaWQgQ0EzMB4XDTI1MDEwNzE3MDg0M1oXDTI1MDIwMjEwMzUyN1owOTEpMCcGA1UEAxMgZDYwMmEwM2E2NzJkODY1YmE1YTQ4NWUzM2EyMDdjNzMxDDAKBgNVBAoTA1RFRTBZMBMGByqGSM49AgEGCCqGSM49AwEHA0IABFPbPYqm91rYvZVCBdFaHRMg0tw7U07JA1EcD9ZP4d0lK2NFM4A0wGKS4jbTR_bu7NTt_YyF388S0PWAJTluqnOjfjB8MB0GA1UdDgQWBBSXyrsZ_A1NnJGRq0sm2G9nm-NC5zAfBgNVHSMEGDAWgBTFUX4F2MtjWykYrAIa8sh9bBL-kjAPBgNVHRMBAf8EBTADAQH_MA4GA1UdDwEB_wQEAwICBDAZBgorBgEEAdZ5AgEeBAuiAQgDZkdvb2dsZTAKBggqhkjOPQQDAgNIADBFAiEAysd6JDoI8X4NEdrRwUwtIAy-hLxSEKUVS2XVWS2CP04CIFNQQzM4TkA_xaZj8KyiS61nb-aOBP35tlA34JCOlv9nWQHcMIIB2DCCAV2gAwIBAgIUAIUK9vrO5iIEbQx0izdwqlWwtk0wCgYIKoZIzj0EAwMwKTETMBEGA1UEChMKR29vZ2xlIExMQzESMBAGA1UEAxMJRHJvaWQgQ0EyMB4XDTI0MTIwOTA2Mjg1M1oXDTI1MDIxNzA2Mjg1MlowKTETMBEGA1UEChMKR29vZ2xlIExMQzESMBAGA1UEAxMJRHJvaWQgQ0EzMFkwEwYHKoZIzj0CAQYIKoZIzj0DAQcDQgAEPjbr-yt9xhgcbKLXoN3RK-1FcCjwIpeMPJZjayW0dqNtFflHp2smO0DxN_6x7M7NAGbcC9lM1_E-N6z51ODv-6NjMGEwDgYDVR0PAQH_BAQDAgIEMA8GA1UdEwEB_wQFMAMBAf8wHQYDVR0OBBYEFMVRfgXYy2NbKRisAhryyH1sEv6SMB8GA1UdIwQYMBaAFKYLhqTwyH8ztWE5Ys0956c6QoNIMAoGCCqGSM49BAMDA2kAMGYCMQCuzU0wV_NkOQzgqzyqP66SJN6lilrU-NDVU6qNCnbFsUoZQOm4wBwUw7LqfoUhx7YCMQDFEvqHfc2hwN2J4I9Z4rTHiLlsy6gA33WvECzIZmVMpKcyEiHlm4c9XR0nVkAjQ_5ZA4QwggOAMIIBaKADAgECAgoDiCZnYGWJloYOMA0GCSqGSIb3DQEBCwUAMBsxGTAXBgNVBAUTEGY5MjAwOWU4NTNiNmIwNDUwHhcNMjIwMTI2MjI0OTQ1WhcNMzcwMTIyMjI0OTQ1WjApMRMwEQYDVQQKEwpHb29nbGUgTExDMRIwEAYDVQQDEwlEcm9pZCBDQTIwdjAQBgcqhkjOPQIBBgUrgQQAIgNiAAT72ZtYJ0I2etFhouvtVs0sBzvYsx8thNCZV1wsDPvsMDSTPij-M1wBFD00OUn2bfU5b7K2_t2NkXc2-_V9g--mdb6SoRGmJ_AG9ScY60LKSA7iPT7gZ_5-q0tnEPPZJCqjZjBkMB0GA1UdDgQWBBSmC4ak8Mh_M7VhOWLNPeenOkKDSDAfBgNVHSMEGDAWgBQ2YeEAfIgFCVGLRGxH_xpMyepPEjASBgNVHRMBAf8ECDAGAQH_AgECMA4GA1UdDwEB_wQEAwIBBjANBgkqhkiG9w0BAQsFAAOCAgEArpB2eLbKHNcS6Q3Td3N7ZCgVLN0qA7CboM-Ftu4YYAcHxh-e_sk7T7XOg5S4d9a_DD7mIXgENSBPB_fVqCnBaSDKNJ3nUuC1_9gcT95p4kKJo0tqcsWw8WgKVJhNuZCN7d_ziHLiRRcrKtaj944THzsy7vB-pSai7gTah_RJrDQI91bDUJgld8_p_QAbVnYA8o-msO0sRKxgF1V5QuBwBTfpdkqshqL3nwBm0sofqI_rM-JOQava3-IurHvfkzioiOJ0uFJnBGVjpZFwGwsmyKwzl-3qRKlkHggAOKt3lQQ4GiJnOCm10JrxPa2Za0K6_kyk6YyvvRcFNai5ej3nMKJPg-eeG2nST6N6ePFuaeoNQnD4XkagGFEQYzcqvsdFsmsbUFMghFl7zEVYdscuSgCG939wxW1JgKyG5ce7CI40328w9IuOf8mUS_W3i4jSfxqCJbegyo_SKDpDILnhJUBy0T3fN8mv9AyO0uoJBlvnogIVv2SdpYUt92vyOiGMy3Jx_ZRWjIRa7iIV3VnjLI__pgCrXQLMinZWEWsxVxg25nrk8u32nZd67DJN3k2FufRbsmHZly9CLo0P79lkIEC3rifLqqJeDyHQNaBMUC6BSDZ5RJCtMjSZw2xL5z0X9_zBsKVPkMW61hMhKzVmYNLe1DJQANRP-enru5i1oXlZBSAwggUcMIIDBKADAgECAgkA1Q_yW6Py1rMwDQYJKoZIhvcNAQELBQAwGzEZMBcGA1UEBRMQZjkyMDA5ZTg1M2I2YjA0NTAeFw0xOTExMjIyMDM3NThaFw0zNDExMTgyMDM3NThaMBsxGTAXBgNVBAUTEGY5MjAwOWU4NTNiNmIwNDUwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIKAoICAQCvtseCK7GnAewrtC6LzFQWY6vvmC8yx391MQMMl1JLG1_oCfvHKqlFH3Q8vZpvEzV0SqVed_a2rDU17hfCXmOVF92ckuY3SlPL_iWPj_u2_RKTeKIqTKmcRS1HpZ8yAfRBl8oczX52L7L1MVG2_rL__Stv5P5bxr2ew0v-CCOdqvzrjrWo7Ss6zZxeOneQ4bUUQnkxWYWYEa2esqlrvdelfJOpHEH8zSfWf9b2caoLgVJhrThPo3lEhkYE3bPYxPkgoZsWVsLxStbQPFbsBgiZBBwe0aX-bTRAtVa60dChUlicU-VdNwdi8BIu75GGGxsObEyAknSZwOm-wLg-O8H5PHLASWBLvS8TReYsP44m2-wGyUdm88EoI51PQxL62BI4h-Br7PVnWDv4NVqB_uq6-ZqDyN8-KjIq_Gcr8SCxNRWLaCHOrzCbbu53-YgzsBjaoQ5FHwajdNUHgfNZCClmu3eLkwiUJpjnTgvNJGKKAcLMA-UfCz5bSsHk356vn_akkqd8FIOIKIUBW0Is5nuAuIybSOE7YHq1Rccj_4xE-PLTaLn2Ug0xFF6_noYq1x32o7_SRQlZ1lN0DZehLzaLE-9m1dClSm4vXZpv70RoMrxnhEclhh8JPdDm80BdqJZD7w9NabZCAFH9uTBJZz42lQWA0830-9CLxYSDlSYAYwIDAQABo2MwYTAdBgNVHQ4EFgQUNmHhAHyIBQlRi0RsR_8aTMnqTxIwHwYDVR0jBBgwFoAUNmHhAHyIBQlRi0RsR_8aTMnqTxIwDwYDVR0TAQH_BAUwAwEB_zAOBgNVHQ8BAf8EBAMCAgQwDQYJKoZIhvcNAQELBQADggIBAE4xoFzyi6Zdva-hztcJae5cqEEErd7YowbPf23uUDdddF7ZkssCQsznLcnu1RGR_lrVK61907JcCZ4TpJGjzdSHpazOh2YyTErkYzgkaue3ikGKy7mKBcTJ1pbuqrYJ0LoM4aMb6YSQ3z9MDqndyegv-w_LPp692MuVJ4nysUEfrFbIhkJutylgQnNdpQ4RrHFfGBjPn9xOJUo3YzUbaiRAFQhhJjpuMQvhpQ3lx-juiA_dS-WISjcSjRiDC7NHa_QpHoLVxmpklJOeCEgL-8APfYp01D5zc36-XY5OxRUwLUaJaSeA3HU47X6Rdb5hOedNQ604izBQ_9Wp3lJiAAiYwB9jxT3-IiCRCPpPZboWxJzL3gg318WETVS3OYugEi5QWxVckxPP4m5y2H4iqhYW5r2_VH3f-T3ynjWmO0Vf4fwOyVWB8_T3u-O7goOWo3rjFXWCvDdkuXgKI578D3Wh4ubZQc6rrCfd6wHivYQhApvqNNUa7mxgJx1alevQBRWpwAE92Av4fuomC4HDT2iObrE0ivDY6hysMqy52T-iSv8DCoTI8rD1acyVCAsgrDWs4MbY29T2hHcZUZ0yRQFm60vxW4WQRFAa3q9DY4LDSxXjtUyS5htpwr_HJkWJFys8k9vjXOBtCP1cATIsoId7HRJ0OvH61ZQOobwC3YkcaGF1dGhEYXRhWMVJlg3liA6MaHQ0Fw9kdmBbj-SuuaKGMseZXPO6gx2XY0UAAAAAuT_ZYfLmRi-xIoIAIkfeeABBAYNe4CBKc8H30FuAb8uaht6JbEQfbSBnS0SX7B6MFg8ofI92oR5lheRDJCgwY-JqB_QSJtezdhMbf8Wzt_La5N2lAQIDJiABIVgg11Yt_p_qwbKz9wOD4_T_HujzYd3jXQt_D2hYgmcjFnUiWCBFQOj2xOvfOT0lAw3J5Nyp56cnOuifxxTrv4HrqolrQA",
		  "clientDataJSON": "eyJ0eXBlIjoid2ViYXV0aG4uY3JlYXRlIiwiY2hhbGxlbmdlIjoidDRMV0kwaVlKU1RXUGw5V1hVZE5oZEhBbnJQRExGOWVXQVA5bEhnbUhQOCIsIm9yaWdpbiI6Imh0dHA6Ly9sb2NhbGhvc3Q6ODAwMCIsImNyb3NzT3JpZ2luIjpmYWxzZX0"
		},
		"type": "public-key",
		"clientExtensionResults": {
		  "credProps": { "rk": false }
		},
		"authenticatorAttachment": "platform"
	  }`,
}
