package mapping

import (
	"encoding/json"
	"strings"
	"testing"
)

func TestMappings(t *testing.T) {
	tagMaps := testData()

	error2string := func(err error) string {
		if err != nil {
			return err.Error()
		}
		return ""
	}

	type FnRes struct{ str, err string }
	type TestCase struct {
		Name     string
		Mappings string
		Results  []FnRes
	}

	cases := []TestCase{
		{"node",
			`{"mappings":[{"tag":"node_name","matcher":"(.*)","template":"$1"}]}`,
			[]FnRes{
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"", "mapping error: missed tag: node_name"},
				{"", "mapping error: missed tag: node_name"},
				{"", "mapping error: missed tag: node_name"},
				{"", "mapping error: missed tag: node_name"},
				{"", "mapping error: missed tag: node_name"},
				{"", "mapping error: missed tag: node_name"},
				{"", "mapping error: missed tag: node_name"},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
				{"minikube", ""},
			},
		},

		{"cluster",
			`{"mappings":[{"tag":"cluster","matcher":"(.*)","template":"$1"}]}`,
			[]FnRes{
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
				{"domain", ""},
			},
		},

		{"cluster-node",
			`{"mappings":[{"tag":"cluster","matcher":"(.*)","template":"$1-"},{"tag":"node_name","matcher":"(.*)","template":"$1"}]}`,
			[]FnRes{
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"", "mapping error: missed tag: node_name"},
				{"", "mapping error: missed tag: node_name"},
				{"", "mapping error: missed tag: node_name"},
				{"", "mapping error: missed tag: node_name"},
				{"", "mapping error: missed tag: node_name"},
				{"", "mapping error: missed tag: node_name"},
				{"", "mapping error: missed tag: node_name"},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
				{"domain-minikube", ""},
			},
		},

		{"container",
			`{"mappings":[{"tag":"container_name","matcher":"(.*)-(.*)","template":"$1-$2"}]}`,
			[]FnRes{
				{"", "mapping error: mismatched tag: container_name"},
				{"", "mapping error: mismatched tag: container_name"},
				{"", "mapping error: mismatched tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"storage-provisioner", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"metrics-server", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"registry-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-controller-manager", ""},
				{"", "mapping error: missed tag: container_name"},
				{"kube-scheduler", ""},
				{"", "mapping error: missed tag: container_name"},
				{"kube-apiserver", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: mismatched tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"calico-node", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"picasa-connector", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: mismatched tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"calico-kube-controllers", ""},
				{"", "mapping error: mismatched tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: mismatched tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
			},
		},

		{"pod-container",
			`{"mappings":[{"tag":"pod_name","matcher":"(.*)","template":"$1-"},{"tag":"container_name","matcher":"(.*)","template":"$1"}]}`,
			[]FnRes{
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"storage-provisioner-storage-provisioner", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"metrics-server-6796b9d575-b26t8-metrics-server", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"registry-proxy-ks9xb-registry-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-controller-manager-minikube-kube-controller-manager", ""},
				{"", "mapping error: missed tag: container_name"},
				{"kube-scheduler-minikube-kube-scheduler", ""},
				{"", "mapping error: missed tag: container_name"},
				{"kube-apiserver-minikube-kube-apiserver", ""},
				{"", "mapping error: missed tag: container_name"},
				{"coredns-bd6b6df9f-hfc84-coredns", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"calico-node-xx6kj-calico-node", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"picasa-connector-wp7hc-picasa-connector", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"etcd-minikube-etcd", ""},
				{"", "mapping error: missed tag: container_name"},
				{"calico-kube-controllers-7b8458594b-chv99-calico-kube-controllers", ""},
				{"registry-t86w2-registry", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"coredns-bd6b6df9f-5pjvl-coredns", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-proxy-kvdm2-kube-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
			},
		},

		{"namespace-container",
			`{"mappings":[{"tag":"namespace","matcher":"(.*)","template":"$1-"},{"tag":"container_name","matcher":"(.*)","template":"$1"}]}`,
			[]FnRes{
				{"", "mapping error: missed tag: namespace"},
				{"", "mapping error: missed tag: namespace"},
				{"", "mapping error: missed tag: namespace"},
				{"", "mapping error: missed tag: namespace"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-storage-provisioner", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-metrics-server", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-registry-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-kube-controller-manager", ""},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-kube-scheduler", ""},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-kube-apiserver", ""},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-coredns", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-calico-node", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-picasa-connector", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-etcd", ""},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-calico-kube-controllers", ""},
				{"kube-system-registry", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-coredns", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-kube-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
			},
		},

		{"namespace-pod-container",
			`{"mappings":[{"tag":"namespace","matcher":"(.*)","template":"$1-"},{"tag":"pod_name","matcher":"(.*)","template":"$1-"},{"tag":"container_name","matcher":"(.*)","template":"$1"}]}`,
			[]FnRes{
				{"", "mapping error: missed tag: namespace"},
				{"", "mapping error: missed tag: namespace"},
				{"", "mapping error: missed tag: namespace"},
				{"", "mapping error: missed tag: namespace"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-storage-provisioner-storage-provisioner", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-metrics-server-6796b9d575-b26t8-metrics-server", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-registry-proxy-ks9xb-registry-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-kube-controller-manager-minikube-kube-controller-manager", ""},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-kube-scheduler-minikube-kube-scheduler", ""},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-kube-apiserver-minikube-kube-apiserver", ""},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-coredns-bd6b6df9f-hfc84-coredns", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-calico-node-xx6kj-calico-node", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-picasa-connector-wp7hc-picasa-connector", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-etcd-minikube-etcd", ""},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-calico-kube-controllers-7b8458594b-chv99-calico-kube-controllers", ""},
				{"kube-system-registry-t86w2-registry", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-coredns-bd6b6df9f-5pjvl-coredns", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"kube-system-kube-proxy-kvdm2-kube-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
			},
		},

		{"tenant-container",
			`{"mappings":[{"tag":"tenant","matcher":"(.*)","template":"$1-"},{"tag":"container_name","matcher":"(.*)","template":"$1"}]}`,
			[]FnRes{
				{"default-kubelet", ""},
				{"default-runtime", ""},
				{"default-pods", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-storage-provisioner", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-metrics-server", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-registry-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-kube-controller-manager", ""},
				{"", "mapping error: missed tag: container_name"},
				{"default-kube-scheduler", ""},
				{"", "mapping error: missed tag: container_name"},
				{"default-kube-apiserver", ""},
				{"", "mapping error: missed tag: container_name"},
				{"default-coredns", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-calico-node", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-picasa-connector", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-etcd", ""},
				{"", "mapping error: missed tag: container_name"},
				{"default-calico-kube-controllers", ""},
				{"default-registry", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-coredns", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-kube-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
			},
		},

		{"tenant-pod-container",
			`{"mappings":[{"tag":"tenant","matcher":"(.*)","template":"$1-"},{"tag":"pod_name","matcher":"(.*)","template":"$1-"},{"tag":"container_name","matcher":"(.*)","template":"$1"}]}`,
			[]FnRes{
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-storage-provisioner-storage-provisioner", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-metrics-server-6796b9d575-b26t8-metrics-server", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-registry-proxy-ks9xb-registry-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-kube-controller-manager-minikube-kube-controller-manager", ""},
				{"", "mapping error: missed tag: container_name"},
				{"default-kube-scheduler-minikube-kube-scheduler", ""},
				{"", "mapping error: missed tag: container_name"},
				{"default-kube-apiserver-minikube-kube-apiserver", ""},
				{"", "mapping error: missed tag: container_name"},
				{"default-coredns-bd6b6df9f-hfc84-coredns", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-calico-node-xx6kj-calico-node", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-picasa-connector-wp7hc-picasa-connector", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-etcd-minikube-etcd", ""},
				{"", "mapping error: missed tag: container_name"},
				{"default-calico-kube-controllers-7b8458594b-chv99-calico-kube-controllers", ""},
				{"default-registry-t86w2-registry", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-coredns-bd6b6df9f-5pjvl-coredns", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"default-kube-proxy-kvdm2-kube-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
			},
		},

		{"label-container",
			`{"mappings":[{"tag":"source","matcher":"^(.*)[|](.*)[|](.*)$","template":"$3-$2-$1-"},{"tag":"container_name","matcher":"(.*)","template":"$1"}]}`,
			[]FnRes{
				{"minikube-domain-default-kubelet", ""},
				{"minikube-domain-default-runtime", ""},
				{"minikube-domain-default-pods", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-storage-provisioner", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-metrics-server", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-registry-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-kube-controller-manager", ""},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-kube-scheduler", ""},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-kube-apiserver", ""},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-coredns", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-calico-node", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-picasa-connector", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-etcd", ""},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-calico-kube-controllers", ""},
				{"minikube-domain-default-registry", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-coredns", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-kube-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
			},
		},

		{"label-pod-container",
			`{"mappings":[{"tag":"source","matcher":"(?m)(?P<name>[[:alnum:]]+)[|](?P<key>[[:alnum:]]+)[|](?P<value>[[:alnum:]]+)$","template":"$value-$key-$name-"},{"tag":"pod_name","matcher":"(.*)","template":"$1-"},{"tag":"container_name","matcher":"(.*)","template":"$1"}]}`,
			[]FnRes{
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: pod_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-storage-provisioner-storage-provisioner", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-metrics-server-6796b9d575-b26t8-metrics-server", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-registry-proxy-ks9xb-registry-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-kube-controller-manager-minikube-kube-controller-manager", ""},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-kube-scheduler-minikube-kube-scheduler", ""},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-kube-apiserver-minikube-kube-apiserver", ""},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-coredns-bd6b6df9f-hfc84-coredns", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-calico-node-xx6kj-calico-node", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-picasa-connector-wp7hc-picasa-connector", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-etcd-minikube-etcd", ""},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-calico-kube-controllers-7b8458594b-chv99-calico-kube-controllers", ""},
				{"minikube-domain-default-registry-t86w2-registry", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-coredns-bd6b6df9f-5pjvl-coredns", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"minikube-domain-default-kube-proxy-kvdm2-kube-proxy", ""},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
				{"", "mapping error: missed tag: container_name"},
			},
		},
	}

	for _, tc := range cases {
		t.Run(tc.Name, func(t *testing.T) {
			p := new(struct {
				Mappings `json:"mappings"`
			})
			if err := json.Unmarshal([]byte(tc.Mappings), p); err != nil {
				t.Errorf("error unmarshalling mappings: %v", err)
			}
			if err := p.Mappings.Compile(); err != nil {
				t.Errorf("error compiling mappings: %v", err)
			}

			for i, tags := range tagMaps {
				str, err := p.Mappings.Apply(tags)
				if tc.Results[i].str != str || tc.Results[i].err != error2string(err) {
					t.Errorf("error:\tnot equal:\n\texpected: %+v\n\tactual  : %+v",
						tc.Results[i], FnRes{str, error2string(err)})
				}

				// t.Logf("%s %d\t %+v", t.Name(), i, FnRes{str, error2string(err)})
			}
		})
	}
}

func TestMappingsApplyOR(t *testing.T) {
	tagMaps := testData()

	error2string := func(err error) string {
		if err != nil {
			return err.Error()
		}
		return ""
	}

	type FnRes struct{ str, err string }
	type TestCase struct {
		Name     string
		Mappings string
		Results  []FnRes
	}

	cases := []TestCase{
		{"pod_or_node",
			`{"mappings":[{"tag":"pod_name","matcher":"(.*)","template":"pod-$1"},{"tag":"node_name","matcher":"(.*)","template":"node-$1"}]}`,
			[]FnRes{
				{"node-minikube", ""},
				{"node-minikube", ""},
				{"node-minikube", ""},
				{"node-minikube", ""},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"pod-metrics-server-6796b9d575-b26t8", ""},
				{"pod-kube-proxy-kvdm2", ""},
				{"pod-coredns-bd6b6df9f-5pjvl", ""},
				{"pod-storage-provisioner", ""},
				{"pod-kube-apiserver-minikube", ""},
				{"pod-kube-scheduler-minikube", ""},
				{"pod-etcd-minikube", ""},
				{"pod-picasa-connector-wp7hc", ""},
				{"pod-coredns-bd6b6df9f-hfc84", ""},
				{"pod-calico-node-xx6kj", ""},
				{"pod-kube-controller-manager-minikube", ""},
				{"pod-calico-kube-controllers-7b8458594b-chv99", ""},
				{"pod-registry-t86w2", ""},
				{"pod-registry-proxy-ks9xb", ""},
				{"pod-storage-provisioner", ""},
				{"pod-storage-provisioner", ""},
				{"pod-storage-provisioner", ""},
				{"pod-metrics-server-6796b9d575-b26t8", ""},
				{"pod-metrics-server-6796b9d575-b26t8", ""},
				{"pod-metrics-server-6796b9d575-b26t8", ""},
				{"pod-metrics-server-6796b9d575-b26t8", ""},
				{"pod-registry-proxy-ks9xb", ""},
				{"pod-registry-proxy-ks9xb", ""},
				{"pod-registry-proxy-ks9xb", ""},
				{"pod-kube-controller-manager-minikube", ""},
				{"pod-kube-controller-manager-minikube", ""},
				{"pod-kube-scheduler-minikube", ""},
				{"pod-kube-scheduler-minikube", ""},
				{"pod-kube-apiserver-minikube", ""},
				{"pod-kube-apiserver-minikube", ""},
				{"pod-coredns-bd6b6df9f-hfc84", ""},
				{"pod-coredns-bd6b6df9f-hfc84", ""},
				{"pod-coredns-bd6b6df9f-hfc84", ""},
				{"pod-coredns-bd6b6df9f-hfc84", ""},
				{"pod-calico-node-xx6kj", ""},
				{"pod-calico-node-xx6kj", ""},
				{"pod-calico-node-xx6kj", ""},
				{"pod-picasa-connector-wp7hc", ""},
				{"pod-picasa-connector-wp7hc", ""},
				{"pod-picasa-connector-wp7hc", ""},
				{"pod-picasa-connector-wp7hc", ""},
				{"pod-etcd-minikube", ""},
				{"pod-etcd-minikube", ""},
				{"pod-calico-kube-controllers-7b8458594b-chv99", ""},
				{"pod-registry-t86w2", ""},
				{"pod-calico-kube-controllers-7b8458594b-chv99", ""},
				{"pod-calico-kube-controllers-7b8458594b-chv99", ""},
				{"pod-registry-t86w2", ""},
				{"pod-registry-t86w2", ""},
				{"pod-coredns-bd6b6df9f-5pjvl", ""},
				{"pod-coredns-bd6b6df9f-5pjvl", ""},
				{"pod-coredns-bd6b6df9f-5pjvl", ""},
				{"pod-coredns-bd6b6df9f-5pjvl", ""},
				{"pod-kube-proxy-kvdm2", ""},
				{"pod-kube-proxy-kvdm2", ""},
				{"pod-kube-proxy-kvdm2", ""},
				{"pod-kube-proxy-kvdm2", ""},
			},
		},

		{"node_pod_or_missing_label",
			`{"mappings":[{"tag":"node_name,pod_name","matcher":"(.*),(.*)","template":"$1-$2"},{"tag":"missing_label","matcher":"(.*)","template":"source-$1"}]}`,
			[]FnRes{
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"", "mapping error: missed tag"},
				{"minikube-metrics-server-6796b9d575-b26t8", ""},
				{"minikube-kube-proxy-kvdm2", ""},
				{"minikube-coredns-bd6b6df9f-5pjvl", ""},
				{"minikube-storage-provisioner", ""},
				{"minikube-kube-apiserver-minikube", ""},
				{"minikube-kube-scheduler-minikube", ""},
				{"minikube-etcd-minikube", ""},
				{"minikube-picasa-connector-wp7hc", ""},
				{"minikube-coredns-bd6b6df9f-hfc84", ""},
				{"minikube-calico-node-xx6kj", ""},
				{"minikube-kube-controller-manager-minikube", ""},
				{"minikube-calico-kube-controllers-7b8458594b-chv99", ""},
				{"minikube-registry-t86w2", ""},
				{"minikube-registry-proxy-ks9xb", ""},
				{"minikube-storage-provisioner", ""},
				{"minikube-storage-provisioner", ""},
				{"minikube-storage-provisioner", ""},
				{"minikube-metrics-server-6796b9d575-b26t8", ""},
				{"minikube-metrics-server-6796b9d575-b26t8", ""},
				{"minikube-metrics-server-6796b9d575-b26t8", ""},
				{"minikube-metrics-server-6796b9d575-b26t8", ""},
				{"minikube-registry-proxy-ks9xb", ""},
				{"minikube-registry-proxy-ks9xb", ""},
				{"minikube-registry-proxy-ks9xb", ""},
				{"minikube-kube-controller-manager-minikube", ""},
				{"minikube-kube-controller-manager-minikube", ""},
				{"minikube-kube-scheduler-minikube", ""},
				{"minikube-kube-scheduler-minikube", ""},
				{"minikube-kube-apiserver-minikube", ""},
				{"minikube-kube-apiserver-minikube", ""},
				{"minikube-coredns-bd6b6df9f-hfc84", ""},
				{"minikube-coredns-bd6b6df9f-hfc84", ""},
				{"minikube-coredns-bd6b6df9f-hfc84", ""},
				{"minikube-coredns-bd6b6df9f-hfc84", ""},
				{"minikube-calico-node-xx6kj", ""},
				{"minikube-calico-node-xx6kj", ""},
				{"minikube-calico-node-xx6kj", ""},
				{"minikube-picasa-connector-wp7hc", ""},
				{"minikube-picasa-connector-wp7hc", ""},
				{"minikube-picasa-connector-wp7hc", ""},
				{"minikube-picasa-connector-wp7hc", ""},
				{"minikube-etcd-minikube", ""},
				{"minikube-etcd-minikube", ""},
				{"minikube-calico-kube-controllers-7b8458594b-chv99", ""},
				{"minikube-registry-t86w2", ""},
				{"minikube-calico-kube-controllers-7b8458594b-chv99", ""},
				{"minikube-calico-kube-controllers-7b8458594b-chv99", ""},
				{"minikube-registry-t86w2", ""},
				{"minikube-registry-t86w2", ""},
				{"minikube-coredns-bd6b6df9f-5pjvl", ""},
				{"minikube-coredns-bd6b6df9f-5pjvl", ""},
				{"minikube-coredns-bd6b6df9f-5pjvl", ""},
				{"minikube-coredns-bd6b6df9f-5pjvl", ""},
				{"minikube-kube-proxy-kvdm2", ""},
				{"minikube-kube-proxy-kvdm2", ""},
				{"minikube-kube-proxy-kvdm2", ""},
				{"minikube-kube-proxy-kvdm2", ""},
			},
		},
	}

	for _, tc := range cases {
		t.Run(tc.Name, func(t *testing.T) {
			p := new(struct {
				Mappings `json:"mappings"`
			})
			if err := json.Unmarshal([]byte(tc.Mappings), p); err != nil {
				t.Errorf("error unmarshalling mappings: %v", err)
			}
			if err := p.Mappings.Compile(); err != nil {
				t.Errorf("error compiling mappings: %v", err)
			}

			for i, tags := range tagMaps {
				str, err := p.Mappings.ApplyOR(tags)
				if tc.Results[i].str != str || !strings.HasPrefix(error2string(err), tc.Results[i].err) {
					t.Errorf("error:\tnot equal:\n\texpected: %+v\n\tactual  : %+v",
						tc.Results[i], FnRes{str, error2string(err)})
				}

				// t.Logf("%s %d\t %+v", t.Name(), i, FnRes{str, error2string(err)})
			}
		})
	}
}

func testData() (tagMaps []map[string]string) {
	payloads := []string{
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"c80cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743631","version":"1.0.0"},"metrics":[{"Name":"kubernetes_system_container","Tags":[{"Key":"cluster","Value":"domain"},{"Key":"container_name","Value":"kubelet"},{"Key":"node_name","Value":"minikube"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"rootfs_available_bytes","Value":0},{"Key":"logsfs_available_bytes","Value":0},{"Key":"cpu_usage_nanocores","Value":79022110},{"Key":"cpu_usage_core_nanoseconds","Value":66139307000},{"Key":"memory_usage_bytes","Value":225746944},{"Key":"memory_rss_bytes","Value":45989888},{"Key":"logsfs_capacity_bytes","Value":0},{"Key":"memory_working_set_bytes","Value":91303936},{"Key":"memory_page_faults","Value":740115},{"Key":"memory_major_page_faults","Value":48},{"Key":"rootfs_capacity_bytes","Value":0}],"Tm":"2023-04-28T08:15:43.631Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"c90cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743631","version":"1.0.0"},"metrics":[{"Name":"kubernetes_system_container","Tags":[{"Key":"cluster","Value":"domain"},{"Key":"container_name","Value":"runtime"},{"Key":"node_name","Value":"minikube"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"cpu_usage_nanocores","Value":31148287},{"Key":"cpu_usage_core_nanoseconds","Value":35762661000},{"Key":"memory_usage_bytes","Value":179613696},{"Key":"memory_rss_bytes","Value":55353344},{"Key":"rootfs_capacity_bytes","Value":0},{"Key":"logsfs_capacity_bytes","Value":0},{"Key":"memory_working_set_bytes","Value":162217984},{"Key":"memory_page_faults","Value":291050},{"Key":"memory_major_page_faults","Value":0},{"Key":"rootfs_available_bytes","Value":0},{"Key":"logsfs_available_bytes","Value":0}],"Tm":"2023-04-28T08:15:43.632Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"ca0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743631","version":"1.0.0"},"metrics":[{"Name":"kubernetes_system_container","Tags":[{"Key":"cluster","Value":"domain"},{"Key":"container_name","Value":"pods"},{"Key":"node_name","Value":"minikube"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"cpu_usage_nanocores","Value":171464110},{"Key":"cpu_usage_core_nanoseconds","Value":203954105000},{"Key":"memory_working_set_bytes","Value":861962240},{"Key":"rootfs_available_bytes","Value":0},{"Key":"rootfs_capacity_bytes","Value":0},{"Key":"logsfs_available_bytes","Value":0},{"Key":"logsfs_capacity_bytes","Value":0},{"Key":"memory_usage_bytes","Value":1942425600},{"Key":"memory_rss_bytes","Value":649957376},{"Key":"memory_page_faults","Value":1630059},{"Key":"memory_major_page_faults","Value":308}],"Tm":"2023-04-28T08:15:43.632Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"cb0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743661","version":"1.0.0"},"metrics":[{"Name":"kubernetes_node","Tags":[{"Key":"cluster","Value":"domain"},{"Key":"node_name","Value":"minikube"},{"Key":"node_status_disk_pressure","Value":"false"},{"Key":"node_status_memory_pressure","Value":"false"},{"Key":"node_status_network_unavailable","Value":"false"},{"Key":"node_status_pid_pressure","Value":"false"},{"Key":"node_status_ready","Value":"true"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"network_rx_bytes","Value":1416560},{"Key":"network_rx_errors","Value":0},{"Key":"network_tx_bytes","Value":15415647},{"Key":"network_tx_errors","Value":0},{"Key":"runtime_image_fs_capacity_bytes","Value":534429827072},{"Key":"runtime_image_fs_used_bytes","Value":1672022278},{"Key":"capacity_memory_bytes","Value":33556598784},{"Key":"memory_working_set_bytes","Value":1498144768},{"Key":"cpu_usage_core_nanoseconds","Value":345906409000},{"Key":"memory_rss_bytes","Value":869601280},{"Key":"memory_page_faults","Value":3871717},{"Key":"fs_available_bytes","Value":116253093888},{"Key":"runtime_image_fs_available_bytes","Value":116253093888},{"Key":"capacity_pods","Value":110},{"Key":"allocatable_cpu_cores","Value":8},{"Key":"cpu_usage_nanocores","Value":298182194},{"Key":"allocatable_millicpu_cores","Value":8000},{"Key":"memory_usage_bytes","Value":2890522624},{"Key":"memory_major_page_faults","Value":686},{"Key":"allocatable_memory_bytes","Value":33556598784},{"Key":"memory_available_bytes","Value":32058454016},{"Key":"fs_used_bytes","Value":414681337856},{"Key":"allocatable_pods","Value":110},{"Key":"capacity_cpu_cores","Value":8},{"Key":"capacity_millicpu_cores","Value":8000},{"Key":"fs_capacity_bytes","Value":534429827072}],"Tm":"2023-04-28T08:15:43.662Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"cc0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743664","version":"1.0.0"},"metrics":[{"Name":"kubernetes_deployment","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"deployment_name","Value":"calico-kube-controllers"},{"Key":"namespace","Value":"kube-system"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"replicas_available","Value":1},{"Key":"replicas_unavailable","Value":0},{"Key":"created","Value":1682413009000000000}],"Tm":"2023-04-28T08:15:43.665Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"cd0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743665","version":"1.0.0"},"metrics":[{"Name":"kubernetes_deployment","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"deployment_name","Value":"coredns"},{"Key":"namespace","Value":"kube-system"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"replicas_available","Value":2},{"Key":"replicas_unavailable","Value":0},{"Key":"created","Value":1682413007000000000}],"Tm":"2023-04-28T08:15:43.665Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"ce0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743665","version":"1.0.0"},"metrics":[{"Name":"kubernetes_deployment","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"deployment_name","Value":"metrics-server"},{"Key":"namespace","Value":"kube-system"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"replicas_available","Value":1},{"Key":"replicas_unavailable","Value":0},{"Key":"created","Value":1682413010000000000}],"Tm":"2023-04-28T08:15:43.666Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"cf0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743666","version":"1.0.0"},"metrics":[{"Name":"kubernetes_daemonset","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"daemonset_name","Value":"picasa-connector"},{"Key":"namespace","Value":"default"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"current_number_scheduled","Value":1},{"Key":"updated_number_scheduled","Value":1},{"Key":"created","Value":1682413129000000000},{"Key":"generation","Value":1},{"Key":"desired_number_scheduled","Value":1},{"Key":"number_available","Value":1},{"Key":"number_misscheduled","Value":0},{"Key":"number_ready","Value":1},{"Key":"number_unavailable","Value":0}],"Tm":"2023-04-28T08:15:43.667Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"d00cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743667","version":"1.0.0"},"metrics":[{"Name":"kubernetes_daemonset","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"daemonset_name","Value":"calico-node"},{"Key":"namespace","Value":"kube-system"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"number_misscheduled","Value":0},{"Key":"number_ready","Value":1},{"Key":"number_unavailable","Value":0},{"Key":"generation","Value":1},{"Key":"current_number_scheduled","Value":1},{"Key":"desired_number_scheduled","Value":1},{"Key":"number_available","Value":1},{"Key":"updated_number_scheduled","Value":1},{"Key":"created","Value":1682413009000000000}],"Tm":"2023-04-28T08:15:43.667Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"d10cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743667","version":"1.0.0"},"metrics":[{"Name":"kubernetes_daemonset","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"daemonset_name","Value":"kube-proxy"},{"Key":"namespace","Value":"kube-system"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"number_available","Value":1},{"Key":"created","Value":1682413007000000000},{"Key":"number_ready","Value":1},{"Key":"number_unavailable","Value":0},{"Key":"updated_number_scheduled","Value":1},{"Key":"generation","Value":1},{"Key":"current_number_scheduled","Value":1},{"Key":"desired_number_scheduled","Value":1},{"Key":"number_misscheduled","Value":0}],"Tm":"2023-04-28T08:15:43.668Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"d20cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743668","version":"1.0.0"},"metrics":[{"Name":"kubernetes_daemonset","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"daemonset_name","Value":"registry-proxy"},{"Key":"namespace","Value":"kube-system"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"generation","Value":1},{"Key":"number_available","Value":1},{"Key":"number_ready","Value":1},{"Key":"number_unavailable","Value":0},{"Key":"created","Value":1682413010000000000},{"Key":"current_number_scheduled","Value":1},{"Key":"desired_number_scheduled","Value":1},{"Key":"number_misscheduled","Value":0},{"Key":"updated_number_scheduled","Value":1}],"Tm":"2023-04-28T08:15:43.668Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"d40cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743782","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"metrics-server-6796b9d575-b26t8"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"pod_status_message","Value":""}],"Tm":"2023-04-28T08:15:43.782Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"d30cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743782","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"kube-proxy-kvdm2"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"pod_status_message","Value":""}],"Tm":"2023-04-28T08:15:43.782Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"d50cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743782","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"coredns-bd6b6df9f-5pjvl"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"pod_status_message","Value":""}],"Tm":"2023-04-28T08:15:43.782Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"d70cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743782","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"storage-provisioner"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"pod_status_message","Value":""}],"Tm":"2023-04-28T08:15:43.783Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"d60cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743782","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"kube-apiserver-minikube"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"pod_status_message","Value":""}],"Tm":"2023-04-28T08:15:43.782Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"d90cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743782","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"kube-scheduler-minikube"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"pod_status_message","Value":""}],"Tm":"2023-04-28T08:15:43.783Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"d80cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743782","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"etcd-minikube"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"pod_status_message","Value":""}],"Tm":"2023-04-28T08:15:43.783Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"db0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743782","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"namespace","Value":"default"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"picasa-connector-wp7hc"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"pod_status_message","Value":""}],"Tm":"2023-04-28T08:15:43.783Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"da0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743782","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"coredns-bd6b6df9f-hfc84"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"pod_status_message","Value":""}],"Tm":"2023-04-28T08:15:43.783Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"dc0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743782","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"calico-node-xx6kj"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"pod_status_message","Value":""}],"Tm":"2023-04-28T08:15:43.783Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"dd0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743782","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"kube-controller-manager-minikube"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"pod_status_message","Value":""}],"Tm":"2023-04-28T08:15:43.783Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"de0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743782","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"calico-kube-controllers-7b8458594b-chv99"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"pod_status_message","Value":""}],"Tm":"2023-04-28T08:15:43.783Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"e00cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743782","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"registry-t86w2"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"pod_status_message","Value":""}],"Tm":"2023-04-28T08:15:43.783Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"df0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743782","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"registry-proxy-ks9xb"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"pod_status_message","Value":""}],"Tm":"2023-04-28T08:15:43.783Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"e10cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743914","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_container","Tags":[{"Key":"addonmanager.kubernetes.io/mode","Value":"Reconcile"},{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"container_name","Value":"storage-provisioner"},{"Key":"integration-test","Value":"storage-provisioner"},{"Key":"kubectl.kubernetes.io/last-applied-configuration","Value":"{\"apiVersion\":\"v1\",\"kind\":\"Pod\",\"metadata\":{\"annotations\":{},\"labels\":{\"addonmanager.kubernetes.io/mode\":\"Reconcile\",\"integration-test\":\"storage-provisioner\"},\"name\":\"storage-provisioner\",\"namespace\":\"kube-system\"},\"spec\":{\"containers\":[{\"command\":[\"/storage-provisioner\"],\"image\":\"gcr.io/k8s-minikube/storage-provisioner:v5\",\"imagePullPolicy\":\"IfNotPresent\",\"name\":\"storage-provisioner\",\"volumeMounts\":[{\"mountPath\":\"/tmp\",\"name\":\"tmp\"}]}],\"hostNetwork\":true,\"serviceAccountName\":\"storage-provisioner\",\"volumes\":[{\"hostPath\":{\"path\":\"/tmp\",\"type\":\"Directory\"},\"name\":\"tmp\"}]}}\n"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237432393Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"phase","Value":"Running"},{"Key":"pod_name","Value":"storage-provisioner"},{"Key":"project","Value":"unknown"},{"Key":"readiness","Value":"ready"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"state","Value":"running"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"rootfs_used_bytes","Value":0},{"Key":"restarts_total","Value":2},{"Key":"container_fs_writes_merged_total","Value":0},{"Key":"container_fs_read_seconds_total","Value":0},{"Key":"cpu_usage_nanocores","Value":3229240},{"Key":"memory_working_set_bytes","Value":8863744},{"Key":"memory_rss_bytes","Value":8216576},{"Key":"rootfs_capacity_bytes","Value":534429827072},{"Key":"container_fs_reads_merged_total","Value":0},{"Key":"container_fs_write_seconds_total","Value":0},{"Key":"memory_usage_bytes","Value":40325120},{"Key":"memory_major_page_faults","Value":10},{"Key":"rootfs_available_bytes","Value":116253093888},{"Key":"logsfs_available_bytes","Value":116253093888},{"Key":"logsfs_used_bytes","Value":16384},{"Key":"container_fs_io_time_seconds_total","Value":0},{"Key":"cpu_usage_core_nanoseconds","Value":1692755000},{"Key":"memory_page_faults","Value":5831},{"Key":"logsfs_capacity_bytes","Value":534429827072},{"Key":"state_code","Value":0},{"Key":"container_fs_writes_total","Value":0},{"Key":"container_fs_reads_total","Value":0}],"Tm":"2023-04-28T08:15:43.914Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"e20cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743916","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"addonmanager.kubernetes.io/mode","Value":"Reconcile"},{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"integration-test","Value":"storage-provisioner"},{"Key":"kubectl.kubernetes.io/last-applied-configuration","Value":"{\"apiVersion\":\"v1\",\"kind\":\"Pod\",\"metadata\":{\"annotations\":{},\"labels\":{\"addonmanager.kubernetes.io/mode\":\"Reconcile\",\"integration-test\":\"storage-provisioner\"},\"name\":\"storage-provisioner\",\"namespace\":\"kube-system\"},\"spec\":{\"containers\":[{\"command\":[\"/storage-provisioner\"],\"image\":\"gcr.io/k8s-minikube/storage-provisioner:v5\",\"imagePullPolicy\":\"IfNotPresent\",\"name\":\"storage-provisioner\",\"volumeMounts\":[{\"mountPath\":\"/tmp\",\"name\":\"tmp\"}]}],\"hostNetwork\":true,\"serviceAccountName\":\"storage-provisioner\",\"volumes\":[{\"hostPath\":{\"path\":\"/tmp\",\"type\":\"Directory\"},\"name\":\"tmp\"}]}}\n"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237432393Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"storage-provisioner"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"kube-api-access-cgwjp"}],"Fields":[{"Key":"used_bytes","Value":12288},{"Key":"available_bytes","Value":33556586496},{"Key":"capacity_bytes","Value":33556598784}],"Tm":"2023-04-28T08:15:43.917Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"e30cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743917","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_network","Tags":[{"Key":"addonmanager.kubernetes.io/mode","Value":"Reconcile"},{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"integration-test","Value":"storage-provisioner"},{"Key":"kubectl.kubernetes.io/last-applied-configuration","Value":"{\"apiVersion\":\"v1\",\"kind\":\"Pod\",\"metadata\":{\"annotations\":{},\"labels\":{\"addonmanager.kubernetes.io/mode\":\"Reconcile\",\"integration-test\":\"storage-provisioner\"},\"name\":\"storage-provisioner\",\"namespace\":\"kube-system\"},\"spec\":{\"containers\":[{\"command\":[\"/storage-provisioner\"],\"image\":\"gcr.io/k8s-minikube/storage-provisioner:v5\",\"imagePullPolicy\":\"IfNotPresent\",\"name\":\"storage-provisioner\",\"volumeMounts\":[{\"mountPath\":\"/tmp\",\"name\":\"tmp\"}]}],\"hostNetwork\":true,\"serviceAccountName\":\"storage-provisioner\",\"volumes\":[{\"hostPath\":{\"path\":\"/tmp\",\"type\":\"Directory\"},\"name\":\"tmp\"}]}}\n"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237432393Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"storage-provisioner"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"tx_bytes","Value":15415647},{"Key":"tx_errors","Value":0},{"Key":"rx_bytes","Value":1416560},{"Key":"rx_errors","Value":0}],"Tm":"2023-04-28T08:15:43.918Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"e40cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743930","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_container","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"4e737ce7ebb60ac2c4a8f3aeccd6109fb83233ea14a115b967c9cccb28f69ba7"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.73/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.73/32"},{"Key":"container_name","Value":"metrics-server"},{"Key":"k8s-app","Value":"metrics-server"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237428826Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"phase","Value":"Running"},{"Key":"pod-template-hash","Value":"6796b9d575"},{"Key":"pod_name","Value":"metrics-server-6796b9d575-b26t8"},{"Key":"project","Value":"unknown"},{"Key":"readiness","Value":"ready"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"state","Value":"running"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"memory_major_page_faults","Value":13},{"Key":"resource_requests_memory_bytes","Value":209715200},{"Key":"container_fs_writes_total","Value":0},{"Key":"container_fs_io_time_seconds_total","Value":0},{"Key":"container_fs_reads_merged_total","Value":0},{"Key":"container_fs_writes_merged_total","Value":0},{"Key":"memory_working_set_bytes","Value":16859136},{"Key":"logsfs_available_bytes","Value":116253093888},{"Key":"restarts_total","Value":1},{"Key":"container_fs_read_seconds_total","Value":0},{"Key":"container_fs_reads_total","Value":0},{"Key":"cpu_usage_nanocores","Value":4490951},{"Key":"memory_usage_bytes","Value":65085440},{"Key":"memory_page_faults","Value":7060},{"Key":"rootfs_used_bytes","Value":0},{"Key":"logsfs_used_bytes","Value":16384},{"Key":"resource_requests_millicpu_units","Value":100},{"Key":"container_fs_write_seconds_total","Value":0},{"Key":"cpu_usage_core_nanoseconds","Value":4001597000},{"Key":"memory_rss_bytes","Value":15912960},{"Key":"rootfs_available_bytes","Value":116253093888},{"Key":"rootfs_capacity_bytes","Value":534429827072},{"Key":"logsfs_capacity_bytes","Value":534429827072},{"Key":"state_code","Value":0}],"Tm":"2023-04-28T08:15:43.93Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"e50cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743932","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"4e737ce7ebb60ac2c4a8f3aeccd6109fb83233ea14a115b967c9cccb28f69ba7"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.73/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.73/32"},{"Key":"k8s-app","Value":"metrics-server"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237428826Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-hash","Value":"6796b9d575"},{"Key":"pod_name","Value":"metrics-server-6796b9d575-b26t8"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"tmp-dir"}],"Fields":[{"Key":"available_bytes","Value":116254748672},{"Key":"capacity_bytes","Value":534429827072},{"Key":"used_bytes","Value":8192}],"Tm":"2023-04-28T08:15:43.933Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"e60cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743933","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"4e737ce7ebb60ac2c4a8f3aeccd6109fb83233ea14a115b967c9cccb28f69ba7"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.73/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.73/32"},{"Key":"k8s-app","Value":"metrics-server"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237428826Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-hash","Value":"6796b9d575"},{"Key":"pod_name","Value":"metrics-server-6796b9d575-b26t8"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"kube-api-access-r2h6d"}],"Fields":[{"Key":"available_bytes","Value":33556586496},{"Key":"capacity_bytes","Value":33556598784},{"Key":"used_bytes","Value":12288}],"Tm":"2023-04-28T08:15:43.934Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"e70cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743934","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_network","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"4e737ce7ebb60ac2c4a8f3aeccd6109fb83233ea14a115b967c9cccb28f69ba7"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.73/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.73/32"},{"Key":"k8s-app","Value":"metrics-server"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237428826Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-hash","Value":"6796b9d575"},{"Key":"pod_name","Value":"metrics-server-6796b9d575-b26t8"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"rx_errors","Value":0},{"Key":"tx_bytes","Value":686705},{"Key":"tx_errors","Value":0},{"Key":"rx_bytes","Value":546756}],"Tm":"2023-04-28T08:15:43.935Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"e80cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743946","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_container","Tags":[{"Key":"addonmanager.kubernetes.io/mode","Value":"Reconcile"},{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"1187a99ffc07ab234cb6dd1213e81bb0d959d6b03ec8c90140e1341e09daab70"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.78/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.78/32"},{"Key":"container_name","Value":"registry-proxy"},{"Key":"controller-revision-hash","Value":"68b8d768bc"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237425701Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"kubernetes.io/minikube-addons","Value":"registry"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"phase","Value":"Running"},{"Key":"pod-template-generation","Value":"1"},{"Key":"pod_name","Value":"registry-proxy-ks9xb"},{"Key":"project","Value":"unknown"},{"Key":"readiness","Value":"ready"},{"Key":"registry-proxy","Value":"true"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"state","Value":"running"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"restarts_total","Value":1},{"Key":"container_fs_writes_total","Value":0},{"Key":"container_fs_read_seconds_total","Value":0},{"Key":"cpu_usage_nanocores","Value":0},{"Key":"memory_working_set_bytes","Value":16150528},{"Key":"memory_major_page_faults","Value":31},{"Key":"rootfs_capacity_bytes","Value":534429827072},{"Key":"logsfs_capacity_bytes","Value":534429827072},{"Key":"rootfs_used_bytes","Value":8192},{"Key":"logsfs_available_bytes","Value":116253093888},{"Key":"container_fs_write_seconds_total","Value":0},{"Key":"cpu_usage_core_nanoseconds","Value":157607000},{"Key":"memory_usage_bytes","Value":23658496},{"Key":"memory_page_faults","Value":4211},{"Key":"container_fs_writes_merged_total","Value":0},{"Key":"container_fs_reads_merged_total","Value":0},{"Key":"container_fs_reads_total","Value":0},{"Key":"memory_rss_bytes","Value":5500928},{"Key":"rootfs_available_bytes","Value":116253093888},{"Key":"logsfs_used_bytes","Value":16384},{"Key":"state_code","Value":0},{"Key":"container_fs_io_time_seconds_total","Value":0}],"Tm":"2023-04-28T08:15:43.946Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"ea0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743954","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"addonmanager.kubernetes.io/mode","Value":"Reconcile"},{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"1187a99ffc07ab234cb6dd1213e81bb0d959d6b03ec8c90140e1341e09daab70"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.78/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.78/32"},{"Key":"controller-revision-hash","Value":"68b8d768bc"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237425701Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"kubernetes.io/minikube-addons","Value":"registry"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-generation","Value":"1"},{"Key":"pod_name","Value":"registry-proxy-ks9xb"},{"Key":"project","Value":"unknown"},{"Key":"registry-proxy","Value":"true"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"kube-api-access-hdz8d"}],"Fields":[{"Key":"capacity_bytes","Value":33556598784},{"Key":"used_bytes","Value":12288},{"Key":"available_bytes","Value":33556586496}],"Tm":"2023-04-28T08:15:43.955Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"ec0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743955","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_network","Tags":[{"Key":"addonmanager.kubernetes.io/mode","Value":"Reconcile"},{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"1187a99ffc07ab234cb6dd1213e81bb0d959d6b03ec8c90140e1341e09daab70"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.78/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.78/32"},{"Key":"controller-revision-hash","Value":"68b8d768bc"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237425701Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"kubernetes.io/minikube-addons","Value":"registry"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-generation","Value":"1"},{"Key":"pod_name","Value":"registry-proxy-ks9xb"},{"Key":"project","Value":"unknown"},{"Key":"registry-proxy","Value":"true"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"tx_errors","Value":0},{"Key":"rx_bytes","Value":4529},{"Key":"rx_errors","Value":0},{"Key":"tx_bytes","Value":2892}],"Tm":"2023-04-28T08:15:43.956Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"eb0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743955","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_container","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"component","Value":"kube-controller-manager"},{"Key":"container_name","Value":"kube-controller-manager"},{"Key":"kubernetes.io/config.hash","Value":"056cc761545f12254730fd15ecec1499"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:53.240051168Z"},{"Key":"kubernetes.io/config.source","Value":"file"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"phase","Value":"Running"},{"Key":"pod_name","Value":"kube-controller-manager-minikube"},{"Key":"project","Value":"unknown"},{"Key":"readiness","Value":"ready"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"state","Value":"running"},{"Key":"tenant","Value":"default"},{"Key":"tier","Value":"control-plane"}],"Fields":[{"Key":"memory_page_faults","Value":17595},{"Key":"logsfs_used_bytes","Value":57344},{"Key":"container_fs_writes_total","Value":0},{"Key":"container_fs_read_seconds_total","Value":0},{"Key":"container_fs_write_seconds_total","Value":0},{"Key":"resource_requests_millicpu_units","Value":200},{"Key":"cpu_usage_nanocores","Value":21059984},{"Key":"cpu_usage_core_nanoseconds","Value":17619443000},{"Key":"memory_usage_bytes","Value":161746944},{"Key":"rootfs_capacity_bytes","Value":534429827072},{"Key":"rootfs_used_bytes","Value":0},{"Key":"logsfs_available_bytes","Value":116253093888},{"Key":"logsfs_capacity_bytes","Value":534429827072},{"Key":"container_fs_io_time_seconds_total","Value":0},{"Key":"container_fs_reads_total","Value":0},{"Key":"memory_rss_bytes","Value":47509504},{"Key":"memory_major_page_faults","Value":30},{"Key":"restarts_total","Value":1},{"Key":"state_code","Value":0},{"Key":"memory_working_set_bytes","Value":48930816},{"Key":"rootfs_available_bytes","Value":116253093888},{"Key":"container_fs_writes_merged_total","Value":0},{"Key":"container_fs_reads_merged_total","Value":0}],"Tm":"2023-04-28T08:15:43.956Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"ed0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743957","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_network","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"component","Value":"kube-controller-manager"},{"Key":"kubernetes.io/config.hash","Value":"056cc761545f12254730fd15ecec1499"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:53.240051168Z"},{"Key":"kubernetes.io/config.source","Value":"file"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"kube-controller-manager-minikube"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"tier","Value":"control-plane"}],"Fields":[{"Key":"tx_errors","Value":0},{"Key":"rx_bytes","Value":1416560},{"Key":"rx_errors","Value":0},{"Key":"tx_bytes","Value":15415647}],"Tm":"2023-04-28T08:15:43.958Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"e90cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743953","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_container","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"component","Value":"kube-scheduler"},{"Key":"container_name","Value":"kube-scheduler"},{"Key":"kubernetes.io/config.hash","Value":"79b5d0893dfbe9c6dca37871e78a9e3b"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:53.240052273Z"},{"Key":"kubernetes.io/config.source","Value":"file"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"phase","Value":"Running"},{"Key":"pod_name","Value":"kube-scheduler-minikube"},{"Key":"project","Value":"unknown"},{"Key":"readiness","Value":"ready"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"state","Value":"running"},{"Key":"tenant","Value":"default"},{"Key":"tier","Value":"control-plane"}],"Fields":[{"Key":"container_fs_writes_merged_total","Value":0},{"Key":"cpu_usage_core_nanoseconds","Value":3568460000},{"Key":"memory_usage_bytes","Value":65769472},{"Key":"rootfs_available_bytes","Value":116253093888},{"Key":"logsfs_capacity_bytes","Value":534429827072},{"Key":"restarts_total","Value":1},{"Key":"resource_requests_millicpu_units","Value":100},{"Key":"container_fs_reads_total","Value":0},{"Key":"container_fs_writes_total","Value":0},{"Key":"memory_rss_bytes","Value":15265792},{"Key":"memory_page_faults","Value":6955},{"Key":"memory_major_page_faults","Value":13},{"Key":"logsfs_used_bytes","Value":20480},{"Key":"state_code","Value":0},{"Key":"container_fs_reads_merged_total","Value":0},{"Key":"memory_working_set_bytes","Value":16121856},{"Key":"rootfs_used_bytes","Value":0},{"Key":"logsfs_available_bytes","Value":116253093888},{"Key":"cpu_usage_nanocores","Value":2888268},{"Key":"rootfs_capacity_bytes","Value":534429827072},{"Key":"container_fs_read_seconds_total","Value":0},{"Key":"container_fs_write_seconds_total","Value":0},{"Key":"container_fs_io_time_seconds_total","Value":0}],"Tm":"2023-04-28T08:15:43.954Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"ee0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743961","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_network","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"component","Value":"kube-scheduler"},{"Key":"kubernetes.io/config.hash","Value":"79b5d0893dfbe9c6dca37871e78a9e3b"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:53.240052273Z"},{"Key":"kubernetes.io/config.source","Value":"file"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"kube-scheduler-minikube"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"tier","Value":"control-plane"}],"Fields":[{"Key":"rx_errors","Value":0},{"Key":"tx_bytes","Value":15415647},{"Key":"tx_errors","Value":0},{"Key":"rx_bytes","Value":1416560}],"Tm":"2023-04-28T08:15:43.961Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"ef0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743969","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_container","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"component","Value":"kube-apiserver"},{"Key":"container_name","Value":"kube-apiserver"},{"Key":"kubeadm.kubernetes.io/kube-apiserver.advertise-address.endpoint","Value":"192.168.49.2:6443"},{"Key":"kubernetes.io/config.hash","Value":"f0480b987ca48bae1e14afb2afebdaf1"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:53.240049640Z"},{"Key":"kubernetes.io/config.source","Value":"file"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"phase","Value":"Running"},{"Key":"pod_name","Value":"kube-apiserver-minikube"},{"Key":"project","Value":"unknown"},{"Key":"readiness","Value":"ready"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"state","Value":"running"},{"Key":"tenant","Value":"default"},{"Key":"tier","Value":"control-plane"}],"Fields":[{"Key":"rootfs_available_bytes","Value":116253093888},{"Key":"logsfs_capacity_bytes","Value":534429827072},{"Key":"restarts_total","Value":1},{"Key":"memory_rss_bytes","Value":260263936},{"Key":"memory_page_faults","Value":76734},{"Key":"rootfs_used_bytes","Value":0},{"Key":"logsfs_available_bytes","Value":116253093888},{"Key":"container_fs_read_seconds_total","Value":0},{"Key":"memory_usage_bytes","Value":388399104},{"Key":"resource_requests_millicpu_units","Value":250},{"Key":"container_fs_io_time_seconds_total","Value":0},{"Key":"container_fs_reads_merged_total","Value":0},{"Key":"container_fs_reads_total","Value":0},{"Key":"container_fs_writes_merged_total","Value":0},{"Key":"memory_major_page_faults","Value":35},{"Key":"cpu_usage_core_nanoseconds","Value":64941386000},{"Key":"memory_working_set_bytes","Value":264773632},{"Key":"rootfs_capacity_bytes","Value":534429827072},{"Key":"logsfs_used_bytes","Value":40960},{"Key":"state_code","Value":0},{"Key":"container_fs_write_seconds_total","Value":0},{"Key":"container_fs_writes_total","Value":0},{"Key":"cpu_usage_nanocores","Value":66914945}],"Tm":"2023-04-28T08:15:43.97Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"f00cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743973","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_network","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"component","Value":"kube-apiserver"},{"Key":"kubeadm.kubernetes.io/kube-apiserver.advertise-address.endpoint","Value":"192.168.49.2:6443"},{"Key":"kubernetes.io/config.hash","Value":"f0480b987ca48bae1e14afb2afebdaf1"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:53.240049640Z"},{"Key":"kubernetes.io/config.source","Value":"file"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"kube-apiserver-minikube"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"tier","Value":"control-plane"}],"Fields":[{"Key":"rx_bytes","Value":1416560},{"Key":"rx_errors","Value":0},{"Key":"tx_bytes","Value":15415647},{"Key":"tx_errors","Value":0}],"Tm":"2023-04-28T08:15:43.974Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"f10cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743979","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_container","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"da093acb3867e93609f32de9f47d8a2ea41d646849b4710054d4ef69ffe7ebcf"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.75/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.75/32"},{"Key":"container_name","Value":"coredns"},{"Key":"k8s-app","Value":"kube-dns"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237433909Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"phase","Value":"Running"},{"Key":"pod-template-hash","Value":"bd6b6df9f"},{"Key":"pod_name","Value":"coredns-bd6b6df9f-hfc84"},{"Key":"project","Value":"unknown"},{"Key":"readiness","Value":"ready"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"state","Value":"running"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"memory_usage_bytes","Value":12857344},{"Key":"memory_major_page_faults","Value":0},{"Key":"logsfs_capacity_bytes","Value":534429827072},{"Key":"logsfs_used_bytes","Value":1314816},{"Key":"state_code","Value":0},{"Key":"container_fs_io_time_seconds_total","Value":0},{"Key":"container_fs_write_seconds_total","Value":0},{"Key":"cpu_usage_nanocores","Value":4017737},{"Key":"memory_working_set_bytes","Value":12857344},{"Key":"memory_rss_bytes","Value":12181504},{"Key":"rootfs_used_bytes","Value":0},{"Key":"resource_requests_millicpu_units","Value":100},{"Key":"memory_page_faults","Value":6919},{"Key":"rootfs_available_bytes","Value":116253093888},{"Key":"logsfs_available_bytes","Value":116253093888},{"Key":"resource_requests_memory_bytes","Value":73400320},{"Key":"resource_limits_memory_bytes","Value":178257920},{"Key":"container_fs_writes_total","Value":0},{"Key":"container_fs_reads_merged_total","Value":0},{"Key":"cpu_usage_core_nanoseconds","Value":3901832000},{"Key":"rootfs_capacity_bytes","Value":534429827072},{"Key":"restarts_total","Value":1},{"Key":"container_fs_read_seconds_total","Value":0},{"Key":"container_fs_reads_total","Value":0},{"Key":"container_fs_writes_merged_total","Value":0}],"Tm":"2023-04-28T08:15:43.98Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"f20cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743980","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"da093acb3867e93609f32de9f47d8a2ea41d646849b4710054d4ef69ffe7ebcf"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.75/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.75/32"},{"Key":"k8s-app","Value":"kube-dns"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237433909Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-hash","Value":"bd6b6df9f"},{"Key":"pod_name","Value":"coredns-bd6b6df9f-hfc84"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"config-volume"}],"Fields":[{"Key":"available_bytes","Value":116254748672},{"Key":"capacity_bytes","Value":534429827072},{"Key":"used_bytes","Value":12288}],"Tm":"2023-04-28T08:15:43.98Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"f30cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743980","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"da093acb3867e93609f32de9f47d8a2ea41d646849b4710054d4ef69ffe7ebcf"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.75/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.75/32"},{"Key":"k8s-app","Value":"kube-dns"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237433909Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-hash","Value":"bd6b6df9f"},{"Key":"pod_name","Value":"coredns-bd6b6df9f-hfc84"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"kube-api-access-bw9sr"}],"Fields":[{"Key":"available_bytes","Value":178245632},{"Key":"capacity_bytes","Value":178257920},{"Key":"used_bytes","Value":12288}],"Tm":"2023-04-28T08:15:43.981Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"f40cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743981","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_network","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"da093acb3867e93609f32de9f47d8a2ea41d646849b4710054d4ef69ffe7ebcf"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.75/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.75/32"},{"Key":"k8s-app","Value":"kube-dns"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237433909Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-hash","Value":"bd6b6df9f"},{"Key":"pod_name","Value":"coredns-bd6b6df9f-hfc84"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"rx_bytes","Value":686672},{"Key":"rx_errors","Value":0},{"Key":"tx_bytes","Value":1030631},{"Key":"tx_errors","Value":0}],"Tm":"2023-04-28T08:15:43.981Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"f50cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743981","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_container","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"container_name","Value":"calico-node"},{"Key":"controller-revision-hash","Value":"b48bf9648"},{"Key":"k8s-app","Value":"calico-node"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237435659Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"phase","Value":"Running"},{"Key":"pod-template-generation","Value":"1"},{"Key":"pod_name","Value":"calico-node-xx6kj"},{"Key":"project","Value":"unknown"},{"Key":"readiness","Value":"ready"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"state","Value":"running"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"memory_usage_bytes","Value":105373696},{"Key":"memory_working_set_bytes","Value":93241344},{"Key":"logsfs_capacity_bytes","Value":534429827072},{"Key":"container_fs_read_seconds_total","Value":0},{"Key":"cpu_usage_core_nanoseconds","Value":35600392000},{"Key":"memory_rss_bytes","Value":74129408},{"Key":"rootfs_used_bytes","Value":159744},{"Key":"container_fs_reads_total","Value":0},{"Key":"memory_major_page_faults","Value":27},{"Key":"rootfs_available_bytes","Value":116253093888},{"Key":"logsfs_available_bytes","Value":116253093888},{"Key":"logsfs_used_bytes","Value":385024},{"Key":"container_fs_write_seconds_total","Value":0},{"Key":"container_fs_writes_total","Value":0},{"Key":"cpu_usage_nanocores","Value":41526816},{"Key":"memory_page_faults","Value":824581},{"Key":"rootfs_capacity_bytes","Value":534429827072},{"Key":"resource_requests_millicpu_units","Value":250},{"Key":"restarts_total","Value":1},{"Key":"state_code","Value":0},{"Key":"container_fs_writes_merged_total","Value":0},{"Key":"container_fs_io_time_seconds_total","Value":0},{"Key":"container_fs_reads_merged_total","Value":0}],"Tm":"2023-04-28T08:15:43.981Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"f60cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743981","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"controller-revision-hash","Value":"b48bf9648"},{"Key":"k8s-app","Value":"calico-node"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237435659Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-generation","Value":"1"},{"Key":"pod_name","Value":"calico-node-xx6kj"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"kube-api-access-zvh6b"}],"Fields":[{"Key":"capacity_bytes","Value":33556598784},{"Key":"used_bytes","Value":12288},{"Key":"available_bytes","Value":33556586496}],"Tm":"2023-04-28T08:15:43.982Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"f70cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743982","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_network","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"controller-revision-hash","Value":"b48bf9648"},{"Key":"k8s-app","Value":"calico-node"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237435659Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-generation","Value":"1"},{"Key":"pod_name","Value":"calico-node-xx6kj"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"rx_bytes","Value":1416560},{"Key":"rx_errors","Value":0},{"Key":"tx_bytes","Value":15415647},{"Key":"tx_errors","Value":0}],"Tm":"2023-04-28T08:15:43.982Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"f80cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743988","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_container","Tags":[{"Key":"app","Value":"unknown"},{"Key":"app.kubernetes.io/name","Value":"picasa-connector"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"c458352994e7855ae5a57f781c1aace7ddabd17b48999982eac751bb8cf080ca"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.76/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.76/32"},{"Key":"container_name","Value":"picasa-connector"},{"Key":"controller-revision-hash","Value":"5b7569c45"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237417615Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"default"},{"Key":"node_name","Value":"minikube"},{"Key":"phase","Value":"Running"},{"Key":"pod-template-generation","Value":"1"},{"Key":"pod_name","Value":"picasa-connector-wp7hc"},{"Key":"project","Value":"unknown"},{"Key":"readiness","Value":"ready"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"state","Value":"running"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"memory_major_page_faults","Value":19},{"Key":"rootfs_available_bytes","Value":116253093888},{"Key":"logsfs_capacity_bytes","Value":534429827072},{"Key":"container_fs_io_time_seconds_total","Value":0},{"Key":"memory_usage_bytes","Value":213458944},{"Key":"memory_working_set_bytes","Value":141209600},{"Key":"rootfs_capacity_bytes","Value":534429827072},{"Key":"logsfs_available_bytes","Value":116253093888},{"Key":"state_code","Value":0},{"Key":"resource_requests_millicpu_units","Value":1000},{"Key":"container_fs_reads_total","Value":0},{"Key":"container_fs_write_seconds_total","Value":0},{"Key":"container_fs_writes_merged_total","Value":0},{"Key":"container_fs_writes_total","Value":0},{"Key":"memory_page_faults","Value":458785},{"Key":"container_fs_reads_merged_total","Value":0},{"Key":"cpu_usage_nanocores","Value":1351126},{"Key":"cpu_usage_core_nanoseconds","Value":44015020000},{"Key":"memory_rss_bytes","Value":135729152},{"Key":"rootfs_used_bytes","Value":2207744},{"Key":"logsfs_used_bytes","Value":15708160},{"Key":"restarts_total","Value":1},{"Key":"resource_requests_memory_bytes","Value":524288000},{"Key":"resource_limits_memory_bytes","Value":838860800},{"Key":"container_fs_read_seconds_total","Value":0}],"Tm":"2023-04-28T08:15:43.988Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"fa0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743989","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"app","Value":"unknown"},{"Key":"app.kubernetes.io/name","Value":"picasa-connector"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"c458352994e7855ae5a57f781c1aace7ddabd17b48999982eac751bb8cf080ca"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.76/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.76/32"},{"Key":"controller-revision-hash","Value":"5b7569c45"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237417615Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"default"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-generation","Value":"1"},{"Key":"pod_name","Value":"picasa-connector-wp7hc"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"config-volume"}],"Fields":[{"Key":"available_bytes","Value":116255862784},{"Key":"capacity_bytes","Value":534429827072},{"Key":"used_bytes","Value":12288}],"Tm":"2023-04-28T08:15:43.989Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"fb0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743989","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"app","Value":"unknown"},{"Key":"app.kubernetes.io/name","Value":"picasa-connector"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"c458352994e7855ae5a57f781c1aace7ddabd17b48999982eac751bb8cf080ca"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.76/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.76/32"},{"Key":"controller-revision-hash","Value":"5b7569c45"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237417615Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"default"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-generation","Value":"1"},{"Key":"pod_name","Value":"picasa-connector-wp7hc"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"kube-api-access-bnsm7"}],"Fields":[{"Key":"available_bytes","Value":838848512},{"Key":"capacity_bytes","Value":838860800},{"Key":"used_bytes","Value":12288}],"Tm":"2023-04-28T08:15:43.99Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"fd0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743990","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_network","Tags":[{"Key":"app","Value":"unknown"},{"Key":"app.kubernetes.io/name","Value":"picasa-connector"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"c458352994e7855ae5a57f781c1aace7ddabd17b48999982eac751bb8cf080ca"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.76/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.76/32"},{"Key":"controller-revision-hash","Value":"5b7569c45"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237417615Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"default"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-generation","Value":"1"},{"Key":"pod_name","Value":"picasa-connector-wp7hc"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"rx_bytes","Value":10800540},{"Key":"rx_errors","Value":0},{"Key":"tx_bytes","Value":1428721},{"Key":"tx_errors","Value":0}],"Tm":"2023-04-28T08:15:43.991Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"f90cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743989","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_container","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"component","Value":"etcd"},{"Key":"container_name","Value":"etcd"},{"Key":"kubeadm.kubernetes.io/etcd.advertise-client-urls","Value":"https://192.168.49.2:2379"},{"Key":"kubernetes.io/config.hash","Value":"86f7fc05a185f3c65295ece1b96005b7"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:53.240035110Z"},{"Key":"kubernetes.io/config.source","Value":"file"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"phase","Value":"Running"},{"Key":"pod_name","Value":"etcd-minikube"},{"Key":"project","Value":"unknown"},{"Key":"readiness","Value":"ready"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"state","Value":"running"},{"Key":"tenant","Value":"default"},{"Key":"tier","Value":"control-plane"}],"Fields":[{"Key":"cpu_usage_core_nanoseconds","Value":18809938000},{"Key":"rootfs_available_bytes","Value":116253093888},{"Key":"restarts_total","Value":1},{"Key":"container_fs_io_time_seconds_total","Value":0},{"Key":"container_fs_writes_merged_total","Value":0},{"Key":"memory_working_set_bytes","Value":102768640},{"Key":"memory_rss_bytes","Value":34271232},{"Key":"memory_major_page_faults","Value":8},{"Key":"rootfs_capacity_bytes","Value":534429827072},{"Key":"state_code","Value":0},{"Key":"container_fs_reads_merged_total","Value":0},{"Key":"container_fs_writes_total","Value":0},{"Key":"rootfs_used_bytes","Value":0},{"Key":"logsfs_capacity_bytes","Value":534429827072},{"Key":"resource_requests_millicpu_units","Value":100},{"Key":"container_fs_read_seconds_total","Value":0},{"Key":"container_fs_reads_total","Value":0},{"Key":"container_fs_write_seconds_total","Value":0},{"Key":"cpu_usage_nanocores","Value":19888629},{"Key":"memory_usage_bytes","Value":143650816},{"Key":"memory_page_faults","Value":11616},{"Key":"logsfs_available_bytes","Value":116253093888},{"Key":"logsfs_used_bytes","Value":28672},{"Key":"resource_requests_memory_bytes","Value":104857600}],"Tm":"2023-04-28T08:15:43.989Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"fe0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743992","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_network","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"component","Value":"etcd"},{"Key":"kubeadm.kubernetes.io/etcd.advertise-client-urls","Value":"https://192.168.49.2:2379"},{"Key":"kubernetes.io/config.hash","Value":"86f7fc05a185f3c65295ece1b96005b7"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:53.240035110Z"},{"Key":"kubernetes.io/config.source","Value":"file"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"etcd-minikube"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"tier","Value":"control-plane"}],"Fields":[{"Key":"rx_errors","Value":0},{"Key":"tx_bytes","Value":15415647},{"Key":"tx_errors","Value":0},{"Key":"rx_bytes","Value":1416560}],"Tm":"2023-04-28T08:15:43.992Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"fc0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743990","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_container","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"1352450e9ff73ec1d3f2632ce05eb16c6f3096a0826735ed6d29fe0dd41b13c9"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.72/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.72/32"},{"Key":"container_name","Value":"calico-kube-controllers"},{"Key":"k8s-app","Value":"calico-kube-controllers"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237389339Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"phase","Value":"Running"},{"Key":"pod-template-hash","Value":"7b8458594b"},{"Key":"pod_name","Value":"calico-kube-controllers-7b8458594b-chv99"},{"Key":"project","Value":"unknown"},{"Key":"readiness","Value":"ready"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"state","Value":"running"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"memory_major_page_faults","Value":18},{"Key":"logsfs_used_bytes","Value":24576},{"Key":"container_fs_writes_merged_total","Value":0},{"Key":"logsfs_available_bytes","Value":116253093888},{"Key":"restarts_total","Value":1},{"Key":"state_code","Value":0},{"Key":"cpu_usage_nanocores","Value":2771294},{"Key":"cpu_usage_core_nanoseconds","Value":2409921000},{"Key":"memory_working_set_bytes","Value":16732160},{"Key":"memory_page_faults","Value":115507},{"Key":"rootfs_used_bytes","Value":4096},{"Key":"container_fs_read_seconds_total","Value":0},{"Key":"memory_usage_bytes","Value":68280320},{"Key":"memory_rss_bytes","Value":12386304},{"Key":"logsfs_capacity_bytes","Value":534429827072},{"Key":"container_fs_write_seconds_total","Value":0},{"Key":"container_fs_writes_total","Value":0},{"Key":"rootfs_available_bytes","Value":116253093888},{"Key":"rootfs_capacity_bytes","Value":534429827072},{"Key":"container_fs_reads_merged_total","Value":0},{"Key":"container_fs_reads_total","Value":0},{"Key":"container_fs_io_time_seconds_total","Value":0}],"Tm":"2023-04-28T08:15:43.99Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"ff0cc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743993","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_container","Tags":[{"Key":"actual-registry","Value":"true"},{"Key":"addonmanager.kubernetes.io/mode","Value":"Reconcile"},{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"ff25a0916fc0ba8b0097487f955083c32f9b20c5fae464bb726ef98700fae0ab"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.77/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.77/32"},{"Key":"container_name","Value":"registry"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237427535Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"kubernetes.io/minikube-addons","Value":"registry"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"phase","Value":"Running"},{"Key":"pod_name","Value":"registry-t86w2"},{"Key":"project","Value":"unknown"},{"Key":"readiness","Value":"ready"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"state","Value":"running"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"cpu_usage_nanocores","Value":88464},{"Key":"memory_page_faults","Value":3135},{"Key":"rootfs_used_bytes","Value":0},{"Key":"container_fs_io_time_seconds_total","Value":0},{"Key":"container_fs_read_seconds_total","Value":0},{"Key":"cpu_usage_core_nanoseconds","Value":285035000},{"Key":"memory_usage_bytes","Value":25657344},{"Key":"memory_working_set_bytes","Value":6107136},{"Key":"memory_major_page_faults","Value":8},{"Key":"rootfs_capacity_bytes","Value":534429827072},{"Key":"logsfs_capacity_bytes","Value":534429827072},{"Key":"memory_rss_bytes","Value":5349376},{"Key":"logsfs_available_bytes","Value":116253093888},{"Key":"restarts_total","Value":1},{"Key":"state_code","Value":0},{"Key":"container_fs_write_seconds_total","Value":0},{"Key":"container_fs_writes_total","Value":0},{"Key":"container_fs_reads_merged_total","Value":0},{"Key":"container_fs_reads_total","Value":0},{"Key":"rootfs_available_bytes","Value":116253093888},{"Key":"logsfs_used_bytes","Value":24576},{"Key":"container_fs_writes_merged_total","Value":0}],"Tm":"2023-04-28T08:15:43.993Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"800dc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743993","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"1352450e9ff73ec1d3f2632ce05eb16c6f3096a0826735ed6d29fe0dd41b13c9"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.72/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.72/32"},{"Key":"k8s-app","Value":"calico-kube-controllers"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237389339Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-hash","Value":"7b8458594b"},{"Key":"pod_name","Value":"calico-kube-controllers-7b8458594b-chv99"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"kube-api-access-bnx4z"}],"Fields":[{"Key":"available_bytes","Value":33556586496},{"Key":"capacity_bytes","Value":33556598784},{"Key":"used_bytes","Value":12288}],"Tm":"2023-04-28T08:15:43.994Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"820dc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743995","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_network","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"1352450e9ff73ec1d3f2632ce05eb16c6f3096a0826735ed6d29fe0dd41b13c9"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.72/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.72/32"},{"Key":"k8s-app","Value":"calico-kube-controllers"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237389339Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-hash","Value":"7b8458594b"},{"Key":"pod_name","Value":"calico-kube-controllers-7b8458594b-chv99"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"tx_bytes","Value":53227},{"Key":"tx_errors","Value":0},{"Key":"rx_bytes","Value":866246},{"Key":"rx_errors","Value":0}],"Tm":"2023-04-28T08:15:43.996Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"810dc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743994","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"actual-registry","Value":"true"},{"Key":"addonmanager.kubernetes.io/mode","Value":"Reconcile"},{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"ff25a0916fc0ba8b0097487f955083c32f9b20c5fae464bb726ef98700fae0ab"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.77/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.77/32"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237427535Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"kubernetes.io/minikube-addons","Value":"registry"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"registry-t86w2"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"kube-api-access-fqrls"}],"Fields":[{"Key":"used_bytes","Value":12288},{"Key":"available_bytes","Value":33556586496},{"Key":"capacity_bytes","Value":33556598784}],"Tm":"2023-04-28T08:15:43.995Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"830dc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669743997","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_network","Tags":[{"Key":"actual-registry","Value":"true"},{"Key":"addonmanager.kubernetes.io/mode","Value":"Reconcile"},{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"ff25a0916fc0ba8b0097487f955083c32f9b20c5fae464bb726ef98700fae0ab"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.77/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.77/32"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237427535Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"kubernetes.io/minikube-addons","Value":"registry"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod_name","Value":"registry-t86w2"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"tx_errors","Value":0},{"Key":"rx_bytes","Value":590},{"Key":"rx_errors","Value":0},{"Key":"tx_bytes","Value":456}],"Tm":"2023-04-28T08:15:43.997Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"840dc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669744000","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_container","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"5cf7a533de921f75630b01ba100921e0a1fb6ca79fca07974eba0cca9f0474bf"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.74/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.74/32"},{"Key":"container_name","Value":"coredns"},{"Key":"k8s-app","Value":"kube-dns"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237421015Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"phase","Value":"Running"},{"Key":"pod-template-hash","Value":"bd6b6df9f"},{"Key":"pod_name","Value":"coredns-bd6b6df9f-5pjvl"},{"Key":"project","Value":"unknown"},{"Key":"readiness","Value":"ready"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"state","Value":"running"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"container_fs_write_seconds_total","Value":0},{"Key":"cpu_usage_nanocores","Value":2191267},{"Key":"memory_working_set_bytes","Value":13303808},{"Key":"memory_rss_bytes","Value":12431360},{"Key":"memory_major_page_faults","Value":14},{"Key":"logsfs_available_bytes","Value":116253093888},{"Key":"logsfs_used_bytes","Value":1339392},{"Key":"resource_limits_memory_bytes","Value":178257920},{"Key":"container_fs_io_time_seconds_total","Value":0},{"Key":"container_fs_reads_total","Value":0},{"Key":"cpu_usage_core_nanoseconds","Value":3902789000},{"Key":"memory_page_faults","Value":6418},{"Key":"rootfs_available_bytes","Value":116253093888},{"Key":"rootfs_used_bytes","Value":0},{"Key":"resource_requests_memory_bytes","Value":73400320},{"Key":"container_fs_writes_merged_total","Value":0},{"Key":"memory_usage_bytes","Value":59805696},{"Key":"resource_requests_millicpu_units","Value":100},{"Key":"container_fs_reads_merged_total","Value":0},{"Key":"rootfs_capacity_bytes","Value":534429827072},{"Key":"logsfs_capacity_bytes","Value":534429827072},{"Key":"restarts_total","Value":1},{"Key":"state_code","Value":0},{"Key":"container_fs_writes_total","Value":0},{"Key":"container_fs_read_seconds_total","Value":0}],"Tm":"2023-04-28T08:15:44.001Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"850dc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669744001","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"5cf7a533de921f75630b01ba100921e0a1fb6ca79fca07974eba0cca9f0474bf"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.74/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.74/32"},{"Key":"k8s-app","Value":"kube-dns"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237421015Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-hash","Value":"bd6b6df9f"},{"Key":"pod_name","Value":"coredns-bd6b6df9f-5pjvl"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"config-volume"}],"Fields":[{"Key":"available_bytes","Value":116254875648},{"Key":"capacity_bytes","Value":534429827072},{"Key":"used_bytes","Value":12288}],"Tm":"2023-04-28T08:15:44.002Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"860dc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669744008","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"5cf7a533de921f75630b01ba100921e0a1fb6ca79fca07974eba0cca9f0474bf"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.74/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.74/32"},{"Key":"k8s-app","Value":"kube-dns"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237421015Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-hash","Value":"bd6b6df9f"},{"Key":"pod_name","Value":"coredns-bd6b6df9f-5pjvl"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"kube-api-access-4df6j"}],"Fields":[{"Key":"available_bytes","Value":178245632},{"Key":"capacity_bytes","Value":178257920},{"Key":"used_bytes","Value":12288}],"Tm":"2023-04-28T08:15:44.008Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"870dc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669744008","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_network","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"cni.projectcalico.org/containerID","Value":"5cf7a533de921f75630b01ba100921e0a1fb6ca79fca07974eba0cca9f0474bf"},{"Key":"cni.projectcalico.org/podIP","Value":"10.244.120.74/32"},{"Key":"cni.projectcalico.org/podIPs","Value":"10.244.120.74/32"},{"Key":"k8s-app","Value":"kube-dns"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237421015Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-hash","Value":"bd6b6df9f"},{"Key":"pod_name","Value":"coredns-bd6b6df9f-5pjvl"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"rx_bytes","Value":700560},{"Key":"rx_errors","Value":0},{"Key":"tx_bytes","Value":1053983},{"Key":"tx_errors","Value":0}],"Tm":"2023-04-28T08:15:44.009Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"880dc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669744016","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_container","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"container_name","Value":"kube-proxy"},{"Key":"controller-revision-hash","Value":"756c5596b8"},{"Key":"k8s-app","Value":"kube-proxy"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237415383Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"phase","Value":"Running"},{"Key":"pod-template-generation","Value":"1"},{"Key":"pod_name","Value":"kube-proxy-kvdm2"},{"Key":"project","Value":"unknown"},{"Key":"readiness","Value":"ready"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"state","Value":"running"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"rootfs_capacity_bytes","Value":534429827072},{"Key":"restarts_total","Value":1},{"Key":"container_fs_reads_total","Value":0},{"Key":"container_fs_write_seconds_total","Value":0},{"Key":"cpu_usage_core_nanoseconds","Value":1407779000},{"Key":"memory_rss_bytes","Value":9662464},{"Key":"memory_page_faults","Value":45486},{"Key":"logsfs_available_bytes","Value":116253093888},{"Key":"logsfs_used_bytes","Value":16384},{"Key":"container_fs_writes_total","Value":0},{"Key":"cpu_usage_nanocores","Value":1678394},{"Key":"memory_usage_bytes","Value":67010560},{"Key":"rootfs_available_bytes","Value":116253093888},{"Key":"rootfs_used_bytes","Value":36864},{"Key":"logsfs_capacity_bytes","Value":534429827072},{"Key":"state_code","Value":0},{"Key":"container_fs_io_time_seconds_total","Value":0},{"Key":"container_fs_read_seconds_total","Value":0},{"Key":"memory_working_set_bytes","Value":17440768},{"Key":"memory_major_page_faults","Value":31},{"Key":"container_fs_reads_merged_total","Value":0},{"Key":"container_fs_writes_merged_total","Value":0}],"Tm":"2023-04-28T08:15:44.016Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"890dc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669744017","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"controller-revision-hash","Value":"756c5596b8"},{"Key":"k8s-app","Value":"kube-proxy"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237415383Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-generation","Value":"1"},{"Key":"pod_name","Value":"kube-proxy-kvdm2"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"kube-proxy"}],"Fields":[{"Key":"available_bytes","Value":116254875648},{"Key":"capacity_bytes","Value":534429827072},{"Key":"used_bytes","Value":20480}],"Tm":"2023-04-28T08:15:44.018Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"8a0dc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669744018","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_volume","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"controller-revision-hash","Value":"756c5596b8"},{"Key":"k8s-app","Value":"kube-proxy"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237415383Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-generation","Value":"1"},{"Key":"pod_name","Value":"kube-proxy-kvdm2"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"},{"Key":"volume_name","Value":"kube-api-access-zvllb"}],"Fields":[{"Key":"available_bytes","Value":33556586496},{"Key":"capacity_bytes","Value":33556598784},{"Key":"used_bytes","Value":12288}],"Tm":"2023-04-28T08:15:44.018Z","Tp":3}]}`,
		`{"context":{"appType":"KUBERNETES","agentId":"Picasa Kubernetes Agent","traceToken":"8b0dc40b-3c34-0886-c7f0-773875a67ff0","timeStamp":"1682669744019","version":"1.0.0"},"metrics":[{"Name":"kubernetes_pod_network","Tags":[{"Key":"app","Value":"unknown"},{"Key":"cluster","Value":"domain"},{"Key":"controller-revision-hash","Value":"756c5596b8"},{"Key":"k8s-app","Value":"kube-proxy"},{"Key":"kubernetes.io/config.seen","Value":"2023-04-28T08:02:58.237415383Z"},{"Key":"kubernetes.io/config.source","Value":"api"},{"Key":"namespace","Value":"kube-system"},{"Key":"node_name","Value":"minikube"},{"Key":"pod-template-generation","Value":"1"},{"Key":"pod_name","Value":"kube-proxy-kvdm2"},{"Key":"project","Value":"unknown"},{"Key":"source","Value":"default|domain|minikube"},{"Key":"tenant","Value":"default"}],"Fields":[{"Key":"tx_errors","Value":0},{"Key":"rx_bytes","Value":1416560},{"Key":"rx_errors","Value":0},{"Key":"tx_bytes","Value":15415647}],"Tm":"2023-04-28T08:15:44.019Z","Tp":3}]}`,
	}

	type P struct {
		Metrics []struct{ Tags []struct{ Key, Value string } }
	}

	tagMaps = make([]map[string]string, 0, len(payloads))
	for _, payload := range payloads {
		p := new(P)
		if err := json.Unmarshal([]byte(payload), p); err != nil {
			panic(err)
		}
		m := map[string]string{}
		for _, tag := range p.Metrics[0].Tags {
			m[tag.Key] = tag.Value
		}
		tagMaps = append(tagMaps, m)
	}
	return tagMaps
}
