// Generate runewidth_table.go from data at https://unicode.org/

package main

import (
	"bufio"
	"bytes"
	"fmt"
	"go/format"
	"io"
	"net/http"
	"os"
	"path"
	"strconv"
	"strings"
)

func fatal(err error) {
	if err != nil {
		fmt.Fprintln(os.Stderr, err.Error())
		os.Exit(1)
	}
}

func get(url string) *os.File {
	file := ".cache/" + path.Base(url)
	fp, err := os.Open(file)
	if err == nil {
		return fp
	}

	resp, err := http.Get(url)
	fatal(err)

	fp, err = os.Create(file)
	fatal(err)

	_, err = io.Copy(fp, resp.Body)
	fatal(err)
	resp.Body.Close()

	return fp
}

func main() {
	err := os.MkdirAll(".cache", 0o755)
	fatal(err)

	east := get("https://www.unicode.org/Public/UCD/latest/ucd/EastAsianWidth.txt")
	emo := get("https://www.unicode.org/Public/UCD/latest/ucd/emoji/emoji-data.txt")

	buf := new(bytes.Buffer)
	fmt.Fprint(buf, "// Code generated by script/generate.go. DO NOT EDIT.\n\n")
	fmt.Fprint(buf, "package runewidth\n\n")

	eastasian(buf, east)
	emoji(buf, emo)

	out, err := format.Source(buf.Bytes())
	fatal(err)

	if len(os.Args) > 1 {
		err = os.WriteFile(os.Args[1]+".go", out, 0o644)
		fatal(err)
	} else {
		fmt.Print(string(out))
	}
}

type rrange struct{ lo, hi rune }

func generate(out io.Writer, v string, arr []rrange) {
	fmt.Fprintf(out, "var %s = table{\n\t", v)
	for i := 0; i < len(arr); i++ {
		fmt.Fprintf(out, "{0x%04X, 0x%04X},", arr[i].lo, arr[i].hi)
		if i < len(arr)-1 {
			if i%3 == 2 {
				fmt.Fprint(out, "\n\t")
			} else {
				fmt.Fprint(out, " ")
			}
		}
	}
	fmt.Fprintln(out, "\n}")
}

func shapeup(p *[]rrange) {
	arr := *p
	for i := 0; i < len(arr)-1; i++ {
		if arr[i].hi+1 == arr[i+1].lo {
			lo := arr[i].lo
			arr = append(arr[:i], arr[i+1:]...)
			arr[i].lo = lo
			i--
		}
	}
	*p = arr
}

func eastasian(out io.Writer, in io.Reader) {
	var (
		dbl, amb, cmb, na, nu []rrange
		scanner               = bufio.NewScanner(in)
	)
	for scanner.Scan() {
		line := scanner.Text()
		if strings.HasPrefix(line, "#") || len(line) == 0 {
			continue
		}

		rng, ss, ok := strings.Cut(line, ";")
		if !ok {
			fatal(fmt.Errorf("; not found in %q", line))
		}

		rs1, rs2, ok := strings.Cut(strings.TrimSpace(rng), "..")
		ri1, err := strconv.ParseInt(rs1, 16, 32)
		fatal(err)
		ri2 := ri1
		if ok {
			ri2, err = strconv.ParseInt(rs2, 16, 32)
			fatal(err)
		}
		r1, r2 := rune(ri1), rune(ri2)

		if strings.Index(line, "COMBINING") != -1 {
			cmb = append(cmb, rrange{lo: r1, hi: r2})
		}

		switch strings.TrimSpace(strings.Fields(ss)[0]) {
		case "W", "F":
			dbl = append(dbl, rrange{lo: r1, hi: r2})
		case "A":
			amb = append(amb, rrange{lo: r1, hi: r2})
		case "Na":
			na = append(na, rrange{lo: r1, hi: r2})
		case "N":
			nu = append(nu, rrange{lo: r1, hi: r2})
		}
	}
	fatal(scanner.Err())

	shapeup(&cmb)
	generate(out, "combining", cmb)
	fmt.Fprintln(out)

	shapeup(&dbl)
	generate(out, "doublewidth", dbl)
	fmt.Fprintln(out)

	shapeup(&amb)
	generate(out, "ambiguous", amb)
	fmt.Fprint(out)

	shapeup(&na)
	generate(out, "narrow", na)
	fmt.Fprintln(out)

	shapeup(&nu)
	generate(out, "neutral", nu)
	fmt.Fprintln(out)
}

func emoji(out io.Writer, in io.Reader) {
	scanner := bufio.NewScanner(in)
	for scanner.Scan() {
		line := scanner.Text()
		if strings.Index(line, "Extended_Pictographic=No") != -1 {
			break
		}
	}
	fatal(scanner.Err())

	var arr []rrange
	for scanner.Scan() {
		line := scanner.Text()
		if strings.HasPrefix(line, "#") || len(line) == 0 {
			continue
		}

		var r1, r2 rune
		n, err := fmt.Sscanf(line, "%x..%x ", &r1, &r2)
		if err != nil || n == 1 {
			n, err = fmt.Sscanf(line, "%x ", &r1)
			if err != nil || n != 1 {
				continue
			}
			r2 = r1
		}
		if r2 < 0xFF {
			continue
		}

		arr = append(arr, rrange{lo: r1, hi: r2})
	}

	shapeup(&arr)
	generate(out, "emoji", arr)
}
