// Copyright 2022-2025 The sacloud/iaas-api-go Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package main

import (
	"fmt"
	"log"
	"path/filepath"

	"github.com/sacloud/iaas-api-go/internal/define"
	"github.com/sacloud/iaas-api-go/internal/dsl"
	"github.com/sacloud/iaas-api-go/internal/tools"
)

const (
	apisDestination = "fake/zz_api_ops.go"
	opsDestination  = "fake/ops_%s.go"
	testDestination = "fake/zz_api_ops_test.go"
)

func init() {
	log.SetFlags(0)
	log.SetPrefix("gen-api-fake-op: ")
}

func main() {
	// generate xxxOp
	outputPath := filepath.Join(tools.ProjectRootPath(), apisDestination)

	tools.WriteFileWithTemplate(&tools.TemplateConfig{
		OutputPath: outputPath,
		Template:   apisTmpl,
		Parameter:  define.APIs,
	})

	log.Printf("generated: %s\n", outputPath)

	// generate funcs
	dsl.IsOutOfSacloudPackage = true
	for _, resource := range define.APIs {
		dest := filepath.Join(tools.ProjectRootPath(), fmt.Sprintf(opsDestination, resource.FileSafeName()))

		wrote := tools.WriteFileWithTemplate(&tools.TemplateConfig{
			OutputPath:         dest,
			Template:           opsTmpl,
			Parameter:          resource,
			PreventOverwriting: true,
		})
		if wrote {
			log.Printf("generated: %s\n", dest)
		}
	}

	// generate xxxOp
	outputPath = filepath.Join(tools.ProjectRootPath(), testDestination)

	tools.WriteFileWithTemplate(&tools.TemplateConfig{
		OutputPath: outputPath,
		Template:   testTmpl,
		Parameter:  define.APIs,
	})
	log.Printf("generated: %s\n", outputPath)
}

const apisTmpl = `// generated by 'github.com/sacloud/iaas-api-go/internal/tools/gen-api-fake-op'; DO NOT EDIT

package fake

import (
{{- range .ImportStatements "sync" "github.com/sacloud/iaas-api-go" "github.com/sacloud/iaas-api-go/types"}}
	{{ . }}
{{- end }}
)

var switchOnce sync.Once

// SwitchFactoryFuncToFake switches iaas.xxxAPI's factory methods to use fake client
func SwitchFactoryFuncToFake() {
	switchOnce.Do(func(){
		switchFactoryFuncToFake()
	})
}

func switchFactoryFuncToFake() {
{{ range . -}}
	iaas.SetClientFactoryFunc(Resource{{.TypeName}}, func(caller iaas.APICaller) interface{} {
		return New{{ .TypeName }}Op()
	})
{{ end -}}
}


{{ range . }}{{ $typeName := .TypeName}} 

/************************************************* 
* {{$typeName}}Op
*************************************************/

// {{ .TypeName }}Op is fake implementation of {{ .TypeName }}API interface
type {{ .TypeName }}Op struct{
	key string
}

// New{{ $typeName}}Op creates new {{ $typeName}}Op instance
func New{{ $typeName}}Op() iaas.{{ $typeName}}API {
	return &{{$typeName}}Op {
		key: Resource{{$typeName}},
	}
}
{{ end -}}
`

const opsTmpl = `package fake

import (
{{- range .ImportStatements "context" "github.com/sacloud/iaas-api-go" "github.com/sacloud/iaas-api-go/types"}}
	{{ . }}
{{- end }}
)

{{ range .Operations }}
// {{ .MethodName }} is fake implementation
func (o *{{ $.TypeName }}Op) {{ .MethodName }}(ctx context.Context{{if not $.IsGlobal}}, zone string{{end}}{{ range .Arguments }}, {{ .ArgName }} {{ .TypeName }}{{ end }}) {{.ResultsStatement}} {
{{ if eq .MethodName "Find" -}}
	results, _ := find(o.key, {{if $.IsGlobal}}iaas.APIDefaultZone{{else}}zone{{end}}, conditions)
	var values []*iaas.{{$.TypeName}}
	for _, res := range results {
		dest := &iaas.{{$.TypeName}}{}
		copySameNameField(res, dest)
		values = append(values, dest)
	}
	return &iaas.{{.ResultTypeName}}{
		Total:    len(results),
		Count:    len(results),
		From:     0,
		{{$.TypeName}}: values,
	}, nil
{{ else if eq .MethodName "List" -}}
	results, _ := find(o.key, {{if $.IsGlobal}}iaas.APIDefaultZone{{else}}zone{{end}}, nil)
	var values []*iaas.{{$.TypeName}}
	for _, res := range results {
		dest := &iaas.{{$.TypeName}}{}
		copySameNameField(res, dest)
		values = append(values, dest)
	}
	return &iaas.{{.ResultTypeName}}{
		Total:    len(results),
		Count:    len(results),
		From:     0,
		{{$.TypeName}}: values,
	}, nil
{{ else if eq .MethodName "Create" -}}
	result := &iaas.{{$.TypeName}}{}
	copySameNameField(param, result)
	fill(result, fillID, fillCreatedAt)

	// TODO core logic is not implemented

	s.set{{$.TypeName}}({{if $.IsGlobal}}iaas.APIDefaultZone{{else}}zone{{end}}, result)
	return result, nil
{{ else if eq .MethodName "Read" -}}
	value := s.get{{$.TypeName}}ByID({{if $.IsGlobal}}iaas.APIDefaultZone{{else}}zone{{end}}, id)
	if value == nil {
		return nil, newErrorNotFound(o.key, id)
	}
	dest := &iaas.{{$.TypeName}}{}
	copySameNameField(value, dest)
	return dest, nil
{{ else if eq .MethodName "Update" -}}
	value, err := o.Read(ctx{{if not $.IsGlobal}}, zone{{end}}, id)
	if err != nil {
		return nil, err
	}
	copySameNameField(param, value)
	fill(value, fillModifiedAt)

	// TODO core logic is not implemented
	return value, nil
{{ else if eq .MethodName "Delete" -}}
	_, err := o.Read(ctx{{if not $.IsGlobal}}, zone{{end}}, id)
	if err != nil {
		return err
	}

	// TODO core logic is not implemented

	s.delete(o.key, {{if $.IsGlobal}}iaas.APIDefaultZone{{else}}zone{{end}}, id)
	return nil
{{ else -}}
	// TODO not implemented
	err := errors.New("not implements")
	return {{.ReturnErrorStatement}}
{{ end -}}
}
{{ end }}
`

const testTmpl = `// generated by 'github.com/sacloud/iaas-api-go/internal/tools/gen-api-fake-op'; DO NOT EDIT

package fake

import (
{{- range .ImportStatements "testing" "github.com/sacloud/iaas-api-go" "github.com/sacloud/iaas-api-go/types"}}
        {{ . }}
{{- end }}
)

func TestResourceOps(t *testing.T) {
{{ range . }}
        if op, ok := New{{.TypeName}}Op().(iaas.{{.TypeName}}API); !ok {
                t.Fatalf("%s is not iaas.{{.TypeName}}", op)
        }
{{ end }}
}`
