package tls

import (
	"errors"
	"fmt"
	"io"
	"testing"
)

func testHandshake(t *testing.T, clientConfig, serverConfig *Config) (serverState, clientState ConnectionState, err error) {
	return testUtlsHandshake(t, clientConfig, serverConfig, spec)
}

func testUtlsHandshake(t *testing.T, clientConfig, serverConfig *Config, spec *ClientHelloSpec) (serverState, clientState ConnectionState, err error) {

	// [uTLS SECTION END]
	const sentinel = "SENTINEL\n"
	c, s := localPipe(t)
	errChan := make(chan error, 1)
	go func() {
		// [uTLS SECTION BEGIN]
		var cli interface {
			Handshake() error
			ConnectionState() ConnectionState
			Close() error
			io.Reader
		}
		if spec != nil {
			ucli := UClient(c, clientConfig, HelloCustom)
			if err = ucli.ApplyPreset(spec); err != nil {
				return
			}
			cli = ucli
		} else {
			cli = Client(c, clientConfig)
		}
		// [uTLS SECTION END]
		err := cli.Handshake()
		if err != nil {
			errChan <- fmt.Errorf("client: %v", err)
			c.Close()
			return
		}

		defer func() { errChan <- nil }()
		clientState = cli.ConnectionState()
		buf, err := io.ReadAll(cli)
		if err != nil {
			t.Errorf("failed to call cli.Read: %v", err)
		}

		if got := string(buf); got != sentinel {
			t.Errorf("read %q from TLS connection, but expected %q", got, sentinel)
		}

		// We discard the error because after ReadAll returns the server must
		// have already closed the connection. Sending data (the closeNotify
		// alert) can cause a reset, that will make Close return an error.
		cli.Close()

	}()

	server := Server(s, serverConfig)
	err = server.Handshake()
	if err == nil {
		serverState = server.ConnectionState()
		if _, err := io.WriteString(server, sentinel); err != nil {
			t.Errorf("failed to call server.Write: %v", err)
		}

		if err := server.Close(); err != nil {
			t.Errorf("failed to call server.Close: %v", err)
		}

	} else {
		err = fmt.Errorf("server: %v", err)
		s.Close()
	}
	err = errors.Join(err, <-errChan)
	return
}

var spec *ClientHelloSpec = nil

func TestDowngradeCanaryUTLS(t *testing.T) {

	chromeLatest, err := utlsIdToSpec(HelloChrome_Auto)
	if err != nil {
		t.Fatal(err)
	}

	firefoxLatest, err := utlsIdToSpec(HelloFirefox_Auto)
	if err != nil {
		t.Fatal(err)
	}

	for _, test := range []struct {
		name          string
		spec          *ClientHelloSpec
		expectSuccess bool
	}{
		{
			name:          "latest chrome",
			spec:          &chromeLatest,
			expectSuccess: true,
		},
		{
			name:          "latest firefox",
			spec:          &firefoxLatest,
			expectSuccess: true,
		},
	} {
		t.Run(test.name, func(t *testing.T) {
			spec = test.spec
			if err := testDowngradeCanary(t, VersionTLS13, VersionTLS12); err == nil {
				t.Errorf("downgrade from TLS 1.3 to TLS 1.2 was not detected")
			}
			if testing.Short() {
				t.Skip("skipping the rest of the checks in short mode")
			}
			if err := testDowngradeCanary(t, VersionTLS13, VersionTLS11); err == nil {
				t.Errorf("downgrade from TLS 1.3 to TLS 1.1 was not detected")
			}
			if err := testDowngradeCanary(t, VersionTLS13, VersionTLS10); err == nil {
				t.Errorf("downgrade from TLS 1.3 to TLS 1.0 was not detected")
			}
			if err := testDowngradeCanary(t, VersionTLS12, VersionTLS11); err == nil {
				t.Errorf("downgrade from TLS 1.2 to TLS 1.1 was not detected")
			}
			if err := testDowngradeCanary(t, VersionTLS12, VersionTLS10); err == nil {
				t.Errorf("downgrade from TLS 1.2 to TLS 1.0 was not detected")
			}
			spec = nil
		})

	}
}
