/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.bmc.auth.internal;

import com.oracle.bmc.InternalSdk;
import com.oracle.bmc.auth.exception.InstancePrincipalUnavailableException;
import com.oracle.bmc.auth.internal.JWK;
import com.oracle.bmc.auth.internal.X509CertificateWithOriginalPem;
import com.oracle.bmc.http.client.Serializer;
import com.oracle.bmc.util.internal.Validate;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.security.KeyFactory;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.X509Certificate;
import java.security.interfaces.RSAPrivateKey;
import java.security.interfaces.RSAPublicKey;
import java.security.spec.RSAPublicKeySpec;
import java.util.Base64;
import java.util.Optional;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.x500.AttributeTypeAndValue;
import org.bouncycastle.asn1.x500.RDN;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.openssl.jcajce.JcaPEMWriter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class AuthUtils {
    private static final char[] HEX_ARRAY = "0123456789ABCDEF".toCharArray();
    private static final Logger LOG = LoggerFactory.getLogger(AuthUtils.class);

    private AuthUtils() {
    }

    public static String getFingerPrint(X509Certificate certificate) {
        Validate.notNull(certificate, "certificate may not be null", new Object[0]);
        try {
            byte[] encodedCertificate = AuthUtils.getEncodedCertificate(certificate);
            MessageDigest md = MessageDigest.getInstance("SHA-256");
            md.update(encodedCertificate);
            String fingerprint = AuthUtils.getHex(md.digest());
            return AuthUtils.formatStringWithSeparator(fingerprint);
        }
        catch (NoSuchAlgorithmException | CertificateEncodingException e) {
            throw new Error(e.getMessage());
        }
    }

    private static String formatStringWithSeparator(String fingerprint) {
        int length = fingerprint.length();
        char[] format = new char[length * 3 / 2 - 1];
        int j = 0;
        for (int i = 0; i < length - 2; i += 2) {
            format[j++] = fingerprint.charAt(i);
            format[j++] = fingerprint.charAt(i + 1);
            format[j++] = 58;
        }
        format[j++] = fingerprint.charAt(length - 2);
        format[j] = fingerprint.charAt(length - 1);
        return String.valueOf(format);
    }

    private static String getHex(byte[] bytes) {
        char[] hexChars = new char[bytes.length * 2];
        for (int j = 0; j < bytes.length; ++j) {
            int v = bytes[j] & 0xFF;
            hexChars[j * 2] = HEX_ARRAY[v >>> 4];
            hexChars[j * 2 + 1] = HEX_ARRAY[v & 0xF];
        }
        return new String(hexChars);
    }

    public static Optional<RSAPublicKey> toPublicKeyFromJson(String json) {
        Validate.notBlank(json, "JSON for public key may not be blank", new Object[0]);
        Optional<JWK> jwk = AuthUtils.toJwk(json);
        if (!jwk.isPresent()) {
            return Optional.empty();
        }
        return AuthUtils.toPublicKeyFromJwk(jwk.get());
    }

    public static Optional<JWK> toJwk(String json) {
        Validate.notBlank(json, "JSON for JWK may not be blank", new Object[0]);
        try {
            JWK jwk = (JWK)Serializer.getDefault().readValue(json, JWK.class);
            return Optional.of(jwk);
        }
        catch (IOException e) {
            LOG.debug("Exception reading or de-serializing jwk", (Throwable)e);
            return Optional.empty();
        }
    }

    public static Optional<RSAPublicKey> toPublicKeyFromJwk(JWK jwk) {
        Validate.notNull(jwk, "JWK may not be null", new Object[0]);
        try {
            RSAPublicKey key = (RSAPublicKey)KeyFactory.getInstance("RSA").generatePublic(new RSAPublicKeySpec(new BigInteger(1, AuthUtils.base64Decode(jwk.getModulus())), new BigInteger(1, AuthUtils.base64Decode(jwk.getPublicExponent()))));
            return Optional.of(key);
        }
        catch (Exception ex) {
            LOG.debug("Failed to construct public key from JWK", (Throwable)ex);
            return Optional.empty();
        }
    }

    @InternalSdk(backwardCompatibilityRequired=true)
    public static byte[] toByteArrayFromRSAPrivateKey(RSAPrivateKey key) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try (JcaPEMWriter writer = new JcaPEMWriter((Writer)new OutputStreamWriter((OutputStream)baos, StandardCharsets.UTF_8));){
            writer.writeObject((Object)key);
            writer.flush();
        }
        catch (IOException e) {
            throw new IllegalStateException("Unable to write PEM object", e);
        }
        return baos.toByteArray();
    }

    public static String base64EncodeNoChunking(RSAPublicKey publicKey) {
        return new String(Base64.getEncoder().encode(publicKey.getEncoded()), StandardCharsets.UTF_8);
    }

    public static String base64EncodeNoChunking(X509Certificate certificate) throws CertificateEncodingException {
        return new String(Base64.getEncoder().encode(AuthUtils.getEncodedCertificate(certificate)), StandardCharsets.UTF_8);
    }

    private static byte[] getEncodedCertificate(X509Certificate certificate) throws CertificateEncodingException {
        if (certificate instanceof X509CertificateWithOriginalPem) {
            return AuthUtils.getEncodedCertificateFromPem(((X509CertificateWithOriginalPem)certificate).getPemEncodedCertificate());
        }
        return certificate.getEncoded();
    }

    public static byte[] base64Decode(String base64) {
        if (base64 == null) {
            return null;
        }
        return Base64.getDecoder().decode(base64.replace('-', '+').replace('_', '/'));
    }

    static byte[] getEncodedCertificateFromPem(String pemEncodedCertificate) {
        return AuthUtils.base64Decode(pemEncodedCertificate.replace("-----BEGIN CERTIFICATE-----", "").replace("-----END CERTIFICATE-----", "").replace("\n", "").replace("\r", ""));
    }

    public static String getTenantIdFromCertificate(X509Certificate certificate) {
        Validate.notNull(certificate, "certificate may not be null", new Object[0]);
        X500Name name = new X500Name(certificate.getSubjectX500Principal().getName());
        Optional<String> tenancyId = AuthUtils.getValue(name, BCStyle.OU, "opc-tenant");
        if (!tenancyId.isPresent()) {
            tenancyId = AuthUtils.getValue(name, BCStyle.O, "opc-identity");
        }
        if (tenancyId.isPresent()) {
            return tenancyId.get();
        }
        throw new InstancePrincipalUnavailableException("The certificate does not contain tenant id.");
    }

    private static Optional<String> getValue(X500Name name, ASN1ObjectIdentifier id, String key) {
        String prefix = key + ":";
        for (RDN rdn : name.getRDNs(id)) {
            for (AttributeTypeAndValue typeAndValue : rdn.getTypesAndValues()) {
                String value = typeAndValue.getValue().toString();
                if (!value.startsWith(prefix)) continue;
                return Optional.of(value.substring(prefix.length()));
            }
        }
        return Optional.empty();
    }
}

