package table

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestTable_sortRows_MissingCells(t *testing.T) {
	table := Table{}
	table.AppendRows([]Row{
		{1, "Arya", "Stark", 3000, 9},
		{11, "Sansa", "Stark", 3000},
		{20, "Jon", "Snow", 2000, "You know nothing, Jon Snow!"},
		{300, "Tyrion", "Lannister", 5000, 7},
	})
	table.SetStyle(StyleDefault)
	table.initForRenderRows()

	// sort by "First Name"
	table.SortBy([]SortBy{{Number: 5, Mode: Asc}})
	assert.Equal(t, []int{1, 3, 0, 2}, table.getSortedRowIndices())
}

func TestTable_sortRows_InvalidMode(t *testing.T) {
	table := Table{}
	table.AppendRows([]Row{
		{1, "Arya", "Stark", 3000},
		{11, "Sansa", "Stark", 3000},
		{20, "Jon", "Snow", 2000, "You know nothing, Jon Snow!"},
		{300, "Tyrion", "Lannister", 5000},
	})
	table.SetStyle(StyleDefault)
	table.initForRenderRows()

	// sort by "First Name"
	table.SortBy([]SortBy{{Number: 2, Mode: AscNumeric}})
	assert.Equal(t, []int{0, 1, 2, 3}, table.getSortedRowIndices())
}

func TestTable_sortRows_MixedMode(t *testing.T) {
	table := Table{}
	table.AppendHeader(Row{"#", "First Name", "Last Name", "Salary"})
	table.AppendRows([]Row{
		/* 0 */ {1, "Arya", "Stark", 3000, 4},
		/* 1 */ {11, "Sansa", "Stark", 3000},
		/* 2 */ {20, "Jon", "Snow", 2000, "You know nothing, Jon Snow!"},
		/* 3 */ {300, "Tyrion", "Lannister", 5000, -7.54},
		/* 4 */ {400, "Jamie", "Lannister", 5000, nil},
		/* 5 */ {500, "Tywin", "Lannister", 5000, "-7.540"},
	})
	table.SetStyle(StyleDefault)
	table.initForRenderRows()

	// sort by nothing
	assert.Equal(t, []int{0, 1, 2, 3, 4, 5}, table.getSortedRowIndices())

	// sort column #5 in Ascending order alphabetically and then numerically
	table.SortBy([]SortBy{{Number: 5, Mode: AscAlphaNumeric}, {Number: 1, Mode: AscNumeric}})
	assert.Equal(t, []int{1, 4, 2, 3, 5, 0}, table.getSortedRowIndices())

	// sort column #5 in Ascending order numerically and then alphabetically
	table.SortBy([]SortBy{{Number: 5, Mode: AscNumericAlpha}, {Number: 1, Mode: AscNumeric}})
	assert.Equal(t, []int{3, 5, 0, 1, 4, 2}, table.getSortedRowIndices())

	// sort column #5 in Descending order alphabetically and then numerically
	table.SortBy([]SortBy{{Number: 5, Mode: DscAlphaNumeric}, {Number: 1, Mode: AscNumeric}})
	assert.Equal(t, []int{2, 4, 1, 0, 3, 5}, table.getSortedRowIndices())

	// sort column #5 in Descending order numerically and then alphabetically
	table.SortBy([]SortBy{{Number: 5, Mode: DscNumericAlpha}, {Number: 1, Mode: AscNumeric}})
	assert.Equal(t, []int{0, 3, 5, 2, 4, 1}, table.getSortedRowIndices())
}

func TestTable_sortRows_WithName(t *testing.T) {
	table := Table{}
	table.AppendHeader(Row{"#", "First Name", "Last Name", "Salary"})
	table.AppendRows([]Row{
		{1, "Arya", "Stark", 3000},
		{11, "Sansa", "Stark", 3000},
		{20, "Jon", "Snow", 2000, "You know nothing, Jon Snow!"},
		{300, "Tyrion", "Lannister", 5000},
	})
	table.SetStyle(StyleDefault)
	table.initForRenderRows()

	// sort by nothing
	assert.Equal(t, []int{0, 1, 2, 3}, table.getSortedRowIndices())

	// sort by "#"
	table.SortBy([]SortBy{{Name: "#", Mode: AscNumeric}})
	assert.Equal(t, []int{0, 1, 2, 3}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Name: "#", Mode: DscNumeric}})
	assert.Equal(t, []int{3, 2, 1, 0}, table.getSortedRowIndices())

	// sort by First Name, Last Name
	table.SortBy([]SortBy{{Name: "First Name", Mode: Asc}, {Name: "Last Name", Mode: Asc}})
	assert.Equal(t, []int{0, 2, 1, 3}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Name: "First Name", Mode: Asc}, {Name: "Last Name", Mode: Dsc}})
	assert.Equal(t, []int{0, 2, 1, 3}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Name: "First Name", Mode: Dsc}, {Name: "Last Name", Mode: Asc}})
	assert.Equal(t, []int{3, 1, 2, 0}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Name: "First Name", Mode: Dsc}, {Name: "Last Name", Mode: Dsc}})
	assert.Equal(t, []int{3, 1, 2, 0}, table.getSortedRowIndices())

	// sort by Last Name, First Name
	table.SortBy([]SortBy{{Name: "Last Name", Mode: Asc}, {Name: "First Name", Mode: Asc}})
	assert.Equal(t, []int{3, 2, 0, 1}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Name: "Last Name", Mode: Asc}, {Name: "First Name", Mode: Dsc}})
	assert.Equal(t, []int{3, 2, 1, 0}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Name: "Last Name", Mode: Dsc}, {Name: "First Name", Mode: Asc}})
	assert.Equal(t, []int{0, 1, 2, 3}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Name: "Last Name", Mode: Dsc}, {Name: "First Name", Mode: Dsc}})
	assert.Equal(t, []int{1, 0, 2, 3}, table.getSortedRowIndices())

	// sort by Unknown Column
	table.SortBy([]SortBy{{Name: "Last Name", Mode: Dsc}, {Name: "Foo Bar", Mode: Dsc}})
	assert.Equal(t, []int{0, 1, 2, 3}, table.getSortedRowIndices())

	// sort by Salary
	table.SortBy([]SortBy{{Name: "Salary", Mode: AscNumeric}})
	assert.Equal(t, []int{2, 0, 1, 3}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Name: "Salary", Mode: DscNumeric}})
	assert.Equal(t, []int{3, 0, 1, 2}, table.getSortedRowIndices())

	table.SortBy(nil)
	assert.Equal(t, []int{0, 1, 2, 3}, table.getSortedRowIndices())
}

func TestTable_sortRows_WithoutName(t *testing.T) {
	table := Table{}
	table.AppendRows([]Row{
		{1, "Arya", "Stark", 3000},
		{11, "Sansa", "Stark", 3000},
		{20, "Jon", "Snow", 2000, "You know nothing, Jon Snow!"},
		{300, "Tyrion", "Lannister", 5000},
	})
	table.SetStyle(StyleDefault)
	table.initForRenderRows()

	// sort by nothing
	assert.Equal(t, []int{0, 1, 2, 3}, table.getSortedRowIndices())

	// sort by "#"
	table.SortBy([]SortBy{{Number: 1, Mode: AscNumeric}})
	assert.Equal(t, []int{0, 1, 2, 3}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Number: 1, Mode: DscNumeric}})
	assert.Equal(t, []int{3, 2, 1, 0}, table.getSortedRowIndices())

	// sort by First Name, Last Name
	table.SortBy([]SortBy{{Number: 2, Mode: Asc}, {Number: 3, Mode: Asc}})
	assert.Equal(t, []int{0, 2, 1, 3}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Number: 2, Mode: Asc}, {Number: 3, Mode: Dsc}})
	assert.Equal(t, []int{0, 2, 1, 3}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Number: 2, Mode: Dsc}, {Number: 3, Mode: Asc}})
	assert.Equal(t, []int{3, 1, 2, 0}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Number: 2, Mode: Dsc}, {Number: 3, Mode: Dsc}})
	assert.Equal(t, []int{3, 1, 2, 0}, table.getSortedRowIndices())

	// sort by Last Name, First Name
	table.SortBy([]SortBy{{Number: 3, Mode: Asc}, {Number: 2, Mode: Asc}})
	assert.Equal(t, []int{3, 2, 0, 1}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Number: 3, Mode: Asc}, {Number: 2, Mode: Dsc}})
	assert.Equal(t, []int{3, 2, 1, 0}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Number: 3, Mode: Dsc}, {Number: 2, Mode: Asc}})
	assert.Equal(t, []int{0, 1, 2, 3}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Number: 3, Mode: Dsc}, {Number: 2, Mode: Dsc}})
	assert.Equal(t, []int{1, 0, 2, 3}, table.getSortedRowIndices())

	// sort by Unknown Column
	table.SortBy([]SortBy{{Number: 3, Mode: Dsc}, {Number: 99, Mode: Dsc}})
	assert.Equal(t, []int{0, 1, 2, 3}, table.getSortedRowIndices())

	// sort by Salary
	table.SortBy([]SortBy{{Number: 4, Mode: AscNumeric}})
	assert.Equal(t, []int{2, 0, 1, 3}, table.getSortedRowIndices())

	table.SortBy([]SortBy{{Number: 4, Mode: DscNumeric}})
	assert.Equal(t, []int{3, 0, 1, 2}, table.getSortedRowIndices())

	table.SortBy(nil)
	assert.Equal(t, []int{0, 1, 2, 3}, table.getSortedRowIndices())
}

func TestTable_sortRows_CustomLess(t *testing.T) {
	t.Run("AllReturnValues", func(t *testing.T) {
		table := Table{}
		table.AppendRows([]Row{
			{"c", "zebra"},
			{"a", "apple"},
			{"b", "banana"},
			{"b", "broccoli"},
		})
		table.initForRenderRows()
		table.SortBy([]SortBy{{
			Number: 1,
			CustomLess: func(iVal string, jVal string) int {
				if iVal < jVal {
					return -1
				}
				if iVal > jVal {
					return 1
				}
				return 0
			},
		}})
		assert.Equal(t, []int{1, 2, 3, 0}, table.getSortedRowIndices())
	})

	t.Run("EqualValuesContinueToNextColumn", func(t *testing.T) {
		table := Table{}
		table.AppendRows([]Row{
			{"same", "zebra"},
			{"same", "apple"},
			{"same", "banana"},
		})
		table.initForRenderRows()
		table.SortBy([]SortBy{
			{
				Number: 1,
				CustomLess: func(iVal string, jVal string) int {
					return 0 // All equal, continue to next column
				},
			},
			{Number: 2, Mode: Asc},
		})
		assert.Equal(t, []int{1, 2, 0}, table.getSortedRowIndices())
	})

	t.Run("WithColumnName", func(t *testing.T) {
		table := Table{}
		table.AppendHeader(Row{"Value", "Other"})
		table.AppendRows([]Row{
			{"zebra", "item"},
			{"apple", "item"},
			{"banana", "item"},
		})
		table.initForRenderRows()
		table.SortBy([]SortBy{{
			Name: "Value",
			CustomLess: func(iVal string, jVal string) int {
				if iVal < jVal {
					return -1
				}
				if iVal > jVal {
					return 1
				}
				return 0
			},
		}})
		assert.Equal(t, []int{1, 2, 0}, table.getSortedRowIndices())
	})
}
