package appsec

import (
	"context"
	"fmt"
	"net/http"

	"github.com/akamai/AkamaiOPEN-edgegrid-golang/v11/pkg/session"
	validation "github.com/go-ozzo/ozzo-validation/v4"
)

type (
	// The MalwareProtection interface supports retrieving and updating malware protection for a configuration and policy.
	// Deprecated: this interface will be removed in a future release. Use the SecurityPolicy interface instead.
	MalwareProtection interface {
		// GetMalwareProtection retrieves the current malware protection setting for a configuration and policy.
		// Deprecated: this method will be removed in a future release. Use the GetPolicyProtections method of the PolicyProtections interface instead.
		//
		// See: https://techdocs.akamai.com/application-security/reference/get-policy-malware
		GetMalwareProtection(ctx context.Context, params GetMalwareProtectionRequest) (*GetMalwareProtectionResponse, error)

		// UpdateMalwareProtection updates the malware protection setting for a configuration and policy.
		// Deprecated: this method will be removed in a future release. Use the CreateSecurityPolicyWithDefaultProtections method of the SecurityPolicy interface instead.
		//
		// See: https://techdocs.akamai.com/application-security/reference/put-policy-malware
		UpdateMalwareProtection(ctx context.Context, params UpdateMalwareProtectionRequest) (*UpdateMalwareProtectionResponse, error)
	}

	// GetMalwareProtectionRequest is used to retrieve the malware protection setting for a policy.
	GetMalwareProtectionRequest struct {
		ConfigID             int    `json:"-"`
		Version              int    `json:"-"`
		PolicyID             string `json:"-"`
		ApplyMalwareControls bool   `json:"applyMalwareControls"`
	}

	// GetMalwareProtectionResponse is returned from a call to GetMalwareProtection.
	GetMalwareProtectionResponse ProtectionsResponse

	// GetMalwareProtectionsRequest is used to retrieve the malware protecton setting for a policy.
	GetMalwareProtectionsRequest struct {
		ConfigID             int    `json:"-"`
		Version              int    `json:"-"`
		PolicyID             string `json:"-"`
		ApplyMalwareControls bool   `json:"applyMalwareControls"`
	}

	// GetMalwareProtectionsResponse is returned from a call to GetMalwareProtections.
	GetMalwareProtectionsResponse ProtectionsResponse

	// UpdateMalwareProtectionRequest is used to modify the malware protection setting.
	UpdateMalwareProtectionRequest struct {
		ConfigID             int    `json:"-"`
		Version              int    `json:"-"`
		PolicyID             string `json:"-"`
		ApplyMalwareControls bool   `json:"applyMalwareControls"`
	}

	// UpdateMalwareProtectionResponse is returned from a call to UpdateMalwareProtection.
	UpdateMalwareProtectionResponse ProtectionsResponse
)

// Validate validates a GetMalwareProtectionRequest.
func (v GetMalwareProtectionRequest) Validate() error {
	return validation.Errors{
		"ConfigID": validation.Validate(v.ConfigID, validation.Required),
		"Version":  validation.Validate(v.Version, validation.Required),
		"PolicyID": validation.Validate(v.PolicyID, validation.Required),
	}.Filter()
}

// Validate validates a GetMalwareProtectionsRequest.
func (v GetMalwareProtectionsRequest) Validate() error {
	return validation.Errors{
		"ConfigID": validation.Validate(v.ConfigID, validation.Required),
		"Version":  validation.Validate(v.Version, validation.Required),
		"PolicyID": validation.Validate(v.PolicyID, validation.Required),
	}.Filter()
}

// Validate validates an UpdateMalwareProtectionRequest.
func (v UpdateMalwareProtectionRequest) Validate() error {
	return validation.Errors{
		"ConfigID": validation.Validate(v.ConfigID, validation.Required),
		"Version":  validation.Validate(v.Version, validation.Required),
		"PolicyID": validation.Validate(v.PolicyID, validation.Required),
	}.Filter()
}

func (p *appsec) GetMalwareProtection(ctx context.Context, params GetMalwareProtectionRequest) (*GetMalwareProtectionResponse, error) {
	logger := p.Log(ctx)
	logger.Debug("GetMalwareProtection")

	if err := params.Validate(); err != nil {
		return nil, fmt.Errorf("%w: %s", ErrStructValidation, err.Error())
	}

	uri := fmt.Sprintf(
		"/appsec/v1/configs/%d/versions/%d/security-policies/%s/protections",
		params.ConfigID,
		params.Version,
		params.PolicyID)

	req, err := http.NewRequestWithContext(ctx, http.MethodGet, uri, nil)
	if err != nil {
		return nil, fmt.Errorf("failed to create GetMalwareProtection request: %w", err)
	}

	var result GetMalwareProtectionResponse
	resp, err := p.Exec(req, &result)
	if err != nil {
		return nil, fmt.Errorf("get malware protection request failed: %w", err)
	}
	defer session.CloseResponseBody(resp)

	if resp.StatusCode != http.StatusOK {
		return nil, p.Error(resp)
	}

	return &result, nil
}

func (p *appsec) GetMalwareProtections(ctx context.Context, params GetMalwareProtectionsRequest) (*GetMalwareProtectionsResponse, error) {
	logger := p.Log(ctx)
	logger.Debug("GetMalwareProtections")

	if err := params.Validate(); err != nil {
		return nil, fmt.Errorf("%w: %s", ErrStructValidation, err.Error())
	}

	uri := fmt.Sprintf(
		"/appsec/v1/configs/%d/versions/%d/security-policies/%s/protections",
		params.ConfigID,
		params.Version,
		params.PolicyID)

	req, err := http.NewRequestWithContext(ctx, http.MethodGet, uri, nil)
	if err != nil {
		return nil, fmt.Errorf("failed to create GetMalwareProtections request: %w", err)
	}

	var result GetMalwareProtectionsResponse
	resp, err := p.Exec(req, &result)
	if err != nil {
		return nil, fmt.Errorf("get malware protections request failed: %w", err)
	}
	defer session.CloseResponseBody(resp)

	if resp.StatusCode != http.StatusOK {
		return nil, p.Error(resp)
	}

	return &result, nil
}

func (p *appsec) UpdateMalwareProtection(ctx context.Context, params UpdateMalwareProtectionRequest) (*UpdateMalwareProtectionResponse, error) {
	logger := p.Log(ctx)
	logger.Debug("UpdateMalwareProtection")

	if err := params.Validate(); err != nil {
		return nil, fmt.Errorf("%w: %s", ErrStructValidation, err.Error())
	}

	uri := fmt.Sprintf(
		"/appsec/v1/configs/%d/versions/%d/security-policies/%s/protections",
		params.ConfigID,
		params.Version,
		params.PolicyID,
	)

	req, err := http.NewRequestWithContext(ctx, http.MethodPut, uri, nil)
	if err != nil {
		return nil, fmt.Errorf("failed to create UpdateMalwareProtection request: %w", err)
	}

	var result UpdateMalwareProtectionResponse
	resp, err := p.Exec(req, &result, params)
	if err != nil {
		return nil, fmt.Errorf("update malware protection request failed: %w", err)
	}
	defer session.CloseResponseBody(resp)

	if resp.StatusCode != http.StatusOK && resp.StatusCode != http.StatusCreated {
		return nil, p.Error(resp)
	}

	return &result, nil
}
