// Copyright (c) 2019 The Jaeger Authors.
// Copyright (c) 2017 Uber Technologies, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
// http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package flags

import (
	"fmt"
	"os"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestParseJaegerTags(t *testing.T) {

	jaegerTags := fmt.Sprintf("%s,%s,%s,%s,%s,%s",
		"key=value",
		"envVar1=${envKey1:defaultVal1}",
		"envVar2=${envKey2:defaultVal2}",
		"envVar3=${envKey3}",
		"envVar4=${envKey4}",
		"envVar5=${envVar5:}",
	)

	os.Setenv("envKey1", "envVal1")
	defer os.Unsetenv("envKey1")

	os.Setenv("envKey4", "envVal4")
	defer os.Unsetenv("envKey4")

	expectedTags := map[string]string{
		"key":     "value",
		"envVar1": "envVal1",
		"envVar2": "defaultVal2",
		"envVar4": "envVal4",
		"envVar5": "",
	}

	assert.Equal(t, expectedTags, ParseJaegerTags(jaegerTags))
}

func TestParseJaegerTagsPanic(t *testing.T) {
	assert.PanicsWithValue(t,
		"invalid Jaeger tag pair \"no-equals-sign\", expected key=value",
		func() {
			ParseJaegerTags("no-equals-sign")
		},
	)
}
