/*
 * PGF: A PGF-codec demonstration
 * $Date: 2006-05-09 20:13:33 +0200 (Di, 09 Mai 2006) $
 * $Revision: 187 $

 * This file Copyright (C) 2006 xeraina GmbH, Switzerland
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

// PGF.cpp : Defines the entry point for the console application.
//
/////////////////////////////////////////////////////////////////////////////
// definitions
#define PGFConsoleVersion	"6.19.3"			// Major number, Minor number: Year (2) Week (2)
#define CurrentYear			"2019"
//-------------------------------------------------------------------------------
// ROI support
//-------------------------------------------------------------------------------
#ifndef NPGFROI
#define __PGFROISUPPORT__ // without ROI support the program code gets simpler and smaller
#endif


#include <iostream>
#include <cmath>	// or #include <math.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctime>
#include <fcntl.h>
#include <string>

#if defined(__linux__) || defined(__APPLE__)
	#define __POSIX__
#endif

#ifdef __POSIX__
	#include <libpgf/PGFimage.h>
	#include <unistd.h>	   // open, close
	#include <stdio.h>
	#include <errno.h>
	#include <string.h>
#else
	#include "PGFimage.h"
	#include <windows.h>
#endif

// must be included after PGFimage.h to avoid type conflicts on POSIX systems
#include "CImage.h"

/////////////////////////////////////////////////////////////////////////////
// definitions
#ifdef __APPLE__
	#define __stat64 stat
	#define _stat64 stat
#elif defined __POSIX__
	#define __stat64 stat64
	#define _stat64 stat64
#endif


using namespace std;

/////////////////////////////////////////////////////////////////////////////
// static variables
static bool bQuiet = false;
static string PGFErrors[] = {
	"no error",
	"memory allocation was not successfull",
	"invalid memory stream position",
	"user break by ESC",
	"wrong pgf version",
	"wrong data file format",
	"image is too small",
	"error in zlib functions",
	"errors related to color table size",
	"errors in png functions",
	"expected data cannot be read",
};

/////////////////////////////////////////////////////////////////////////////
static INT64 FileSize(char *filename) {
	struct __stat64 data;

 	if (_stat64(filename, &data) != -1) {
		return data.st_size;
	} else {
		return 0;
	}
}

/////////////////////////////////////////////////////////////////////////////
static void PSNR(const CImage& image1, const CImage& image2, bool roi, PGFRect& rect) {
#ifndef __PGFROISUPPORT__
	roi;
	rect;
#endif
	ASSERT(image1.GetChannelDepth() == image2.GetChannelDepth());
	ASSERT(image1.GetChannelDepth() == 8 || image1.GetChannelDepth() == 16);

	const int channels1 = image1.GetChannels(); ASSERT(channels1 <= MaxChannels);
	#pragma warning(disable: 4189)
	const int channels2 = image2.GetChannels(); ASSERT(channels2 <= MaxChannels);
	ASSERT(channels1 == channels2);
	const UINT32 w = image2.GetWidth();
	const UINT32 h = image2.GetHeight();
	const UINT32 size = w*h;
	const int bypp1 = image1.GetBPP()/8;	// RGB mode has 3 channels but can use 24 or 32 bits per pixel
	const int bypp2 = image2.GetBPP()/8;

	int pitch1 = image1.GetPitch();
	int pitch2 = image2.GetPitch();
	double sum[MaxChannels + 1] = { 0 };
	int tmp;
	int cnt1 = 0, cnt2 = 0, maxValue;

	if (image1.GetChannelDepth() == 8) {
		UINT8* rgbBuff1 = (UINT8 *)image1.GetBits();
		UINT8* rgbBuff2 = (UINT8 *)image2.GetBits();
		maxValue = 255;

	#ifdef __PGFROISUPPORT__
		if (roi) {
			ASSERT(w <= image1.GetWidth());
			ASSERT(h <= image1.GetHeight());
			rgbBuff1 += (image1.GetHeight() - h - rect.top)*pitch1 + rect.left*channels1;
		}
	#endif
		for (UINT32 j=0; j < h; j++) {
			cnt1 = cnt2 = 0;
			for (UINT32 i=0; i < w; i++) {
				for (int c=0; c < channels1; c++) {
					tmp = rgbBuff2[cnt2 + c] - rgbBuff1[cnt1 + c]; sum[c] += tmp*tmp;
				}
				cnt1 += bypp1;
				cnt2 += bypp2;
			}
			rgbBuff1 += pitch1;
			rgbBuff2 += pitch2;
		}
	} else if (image1.GetChannelDepth() == 16) {
		UINT16* rgbBuff1 = (UINT16 *)image1.GetBits();
		UINT16* rgbBuff2 = (UINT16 *)image2.GetBits();

#ifdef __PNMEXSUPPORT__
		maxValue = image1.GetMaxValue();
#else
		maxValue = 65535;
#endif
		pitch1 /= 2;
		pitch2 /= 2;

	#ifdef __PGFROISUPPORT__
		if (roi) {
			ASSERT(w <= image1.GetWidth());
			ASSERT(h <= image1.GetHeight());
			rgbBuff1 += (image1.GetHeight() - h - rect.top)*pitch1 + rect.left*channels1;
		}
	#endif
		for (UINT32 j=0; j < h; j++) {
			cnt1 = cnt2 = 0;
			for (UINT32 i=0; i < w; i++) {
				for (int c=0; c < channels1; c++) {
					tmp = rgbBuff2[cnt2 + c] - rgbBuff1[cnt1 + c]; sum[c] += tmp*tmp;
				}
				cnt1 += bypp1;
				cnt2 += bypp2;
			}
			rgbBuff1 += pitch1;
			rgbBuff2 += pitch2;
		}
	} else {
		return;
	}

	for (int c=0; c < channels1; c++) {
		sum[MaxChannels] += sum[c];
	}

	// output
	if (bQuiet) {
		cout << 10*log10(double(maxValue)*maxValue*size*channels1/sum[MaxChannels]);
//		cout << ((sum[MaxChannels] != 0) ? 10*log10(double(maxValue)*maxValue*size*channels1/sum[MaxChannels]) : 100);
		for (int c=0; c < channels1; c++) {
			cout << ';' << 10*log10(double(maxValue)*maxValue*size/sum[c]);
			//cout << ';' << ((sum[c] != 0) ? 10*log10(double(maxValue)*maxValue*size/sum[c]) : 100);
		}
		cout << ';';
	} else {
		if (sum[MaxChannels] == 0) {
			cout << "PSNR: lossless" << endl;
		} else {
			cout << "PSNR: " << 10*log10(double(maxValue)*maxValue*size*channels1/sum[MaxChannels]) << " (";
			for (int c=0; c < channels1; c++) {
				cout << 'c' << c << ": " << 10*log10(double(maxValue)*maxValue*size/sum[c]);
				//cout << 'c' << c << ": " << ((sum[c] != 0) ? 10*log10(double(maxValue)*maxValue*size/sum[c]) : 100);
				if (c < channels1 - 1) cout << ", ";
			}
			cout << ')' << endl;
		}
		cout << endl;
	}
}

/////////////////////////////////////////////////////////////////////////////
static bool Encoding(CImage*& image, char *source, char *dest, int levels, int quality, bool roi, bool streaming, CPGFMemoryStream** memStream) {
#ifndef __PGFROISUPPORT__
	ASSERT(!roi);
	ASSERT(!streaming);
#endif
	ASSERT(source);
	ASSERT(0 <= quality && quality <= MaxQuality);
	bool returnValue = true;
	CPGFImage pgf;
	PGFHeader header;
	clock_t start = 0, mean = 0, end = 0;
	UINT32 writtenBytes = 0;
	CPGFStream *stream = nullptr;
	UINT8 bpp = 0;
#ifdef WIN32
	HANDLE fd = nullptr;
#elif defined(__POSIX__)
	int fd = 0;
#endif

	if (!memStream) {
		ASSERT(dest);
	#ifdef WIN32
		// no other choice to get a "HANDLE"
		fd = CreateFile(dest, GENERIC_WRITE, FILE_SHARE_READ, 0, CREATE_ALWAYS, 0, 0);
		if (fd == INVALID_HANDLE_VALUE) {
			cerr << "Error: Could not open destination file." << endl;
			fd = nullptr;
			returnValue = false;
			goto CleanUp;
		}
		
		// create stream object
		stream = new CPGFFileStream(fd);

	#elif defined(__POSIX__)
		fd = open(dest, O_RDWR | O_CREAT | O_TRUNC, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
		if (fd == -1) {
			cout << "Error: Could not open destination file." << endl;
			fd = 0;
			returnValue = false;
			goto CleanUp;
		}
		
		// create stream object
		stream = new CPGFFileStream(fd);
	#endif
	}

	if (memStream) {
		if (!bQuiet) cout << "Encoding PGF image to memory stream (quality: " << quality << ", levels: " << levels << "): " << source << endl;
	} else {
		if (!bQuiet) cout << "Encoding PGF image (quality: " << quality << ", levels: " << levels << "): " << source << endl;
	}
	if (roi || streaming) {
		if (memStream) {
			cout << "PGF image will support ROI." << endl;
		} else {
			cout << dest << " will support ROI." << endl;
		}
	}

	start = clock();

	// create new image object
	image = new CImage();

	// load image
#ifdef WIN32
	if (!image->Load(source)) {
		LPTSTR lpMsgBuf;
		FormatMessage( 
			FORMAT_MESSAGE_ALLOCATE_BUFFER | 
			FORMAT_MESSAGE_FROM_SYSTEM | 
			FORMAT_MESSAGE_IGNORE_INSERTS,
			nullptr,
			GetLastError(),
			MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
			(LPTSTR)&lpMsgBuf,
			0,
			nullptr 
		); 
		cerr << "Error: " << lpMsgBuf << endl;
		LocalFree(lpMsgBuf);
		returnValue = false;
		goto CleanUp;
	}
#elif defined(__POSIX__)
	if (!image->Load(source) ){
		cerr << "Error: Could not load source file." << endl;
		returnValue = false;
		goto CleanUp;
	}
#endif

	bpp = image->GetBPP();
	// minimum: 1-bit bitmap, maximum: 48-bit RGB (16-bit per channel)
	if ((bpp < 1) || (bpp > 48) || (bpp == 4)) {
		cerr << "Error: Unhandled image format." << endl;
		returnValue = false;
		goto CleanUp;
	}

	ASSERT((bpp >= 1) && (bpp <= 48) && (bpp != 4));

	if (memStream) {
		*memStream = new CPGFMemoryStream(abs(image->GetPitch())*image->GetHeight());
		stream = *memStream;
	}

	mean = clock();

	// optional PGF encoder configuration
	pgf.ConfigureEncoder(true, false); // true: use openMP (if codec is compiled with openMP), false: favorSpeedOverSize

	header.width = image->GetWidth();
	header.height = image->GetHeight();
	header.nLevels = (UINT8)levels;
	header.quality = (UINT8)quality;
	header.mode = image->GetColorType();	
	header.bpp = header.channels = 0; // depend on mode and will be set automatically
	header.usedBitsPerChannel = 0; // depend on mode and bpp and will be set automatically

	try {
		if (bQuiet) {
			pgf.SetHeader(header, (roi || streaming) ? PGFROI : 0);
		} else {
			char data[] = "This is a free text annotation.";
			pgf.SetHeader(header, (roi || streaming) ? PGFROI : 0, (UINT8 *)data, sizeof(data));
		}

#ifdef __PNMEXSUPPORT__
		if (image->GetChannelDepth() > 8) {
			// set maximum value
			pgf.SetMaxValue(image->GetMaxValue());
		}
#endif

		// copy bits
		UINT8* buff = (UINT8 *)image->GetBits();
		if (pgf.Mode() == ImageModeRGB48) {
			int map[] = { 2, 1, 0 };
			pgf.ImportBitmap(-image->GetPitch(), &(buff[(image->GetHeight() - 1)*image->GetPitch()]), bpp, map);
		} else {
#ifdef __BIG_ENDIAN__
			int map[] = { 2, 1, 0 };
			pgf.ImportBitmap(-image->GetPitch(), &(buff[(image->GetHeight() - 1)*image->GetPitch()]), bpp, map);
#else
			pgf.ImportBitmap(-image->GetPitch(), &(buff[(image->GetHeight() - 1)*image->GetPitch()]), bpp);
#endif
		}

		// update color table if image is indexed
		if ((pgf.Mode() == ImageModeIndexedColor)) {
			int colorTableSize = image->GetMaxColorTableEntries();
			RGBQUAD* pColorTable = new RGBQUAD[colorTableSize];

			image->GetColorTable(0, colorTableSize, pColorTable);
			pgf.SetColorTable(0, colorTableSize, pColorTable);
			delete[] pColorTable;
		}

	} catch(IOException& e) {
		int err = e.error;

		if (err >= AppError) {
			cerr << "Error: Importing input image failed\n(" << PGFErrors[err - AppError] << ")!" << endl;
		} else {
			cerr << "Error: Importing input image failed (" << err << ")!" << endl;
		}
		returnValue = false;
		goto CleanUp;
	}

	try {
	#ifdef __PGFROISUPPORT__
		if (streaming) {
			// process and write header
			writtenBytes = pgf.WriteHeader(stream);
			if (!bQuiet) cout << "Write header [" << writtenBytes << " bytes]" << endl;
			// encode each level separately
			for(int i = pgf.Levels() - 1; i >= 0; i--) {
				UINT32 size = pgf.Write(i);
				if (!bQuiet) cout << "Write level " << i << " [" << size << " bytes]" << endl;
				writtenBytes += size;
			}
		} else 
	#endif __PGFROISUPPORT__
		{
			// write image to pgf-file
			pgf.Write(stream, &writtenBytes);
		}

	} catch(IOException& e) {
		int err = e.error;
		if (err >= AppError) {
			cerr << "Error: Writing PGF image failed\n(" << PGFErrors[err - AppError] << ")!" << endl;
		} else {
			cerr << "Error: Writing PGF image failed (" << err << ")!" << endl;
		}
		returnValue = false;
		goto CleanUp;
	}

	end = clock();
		
CleanUp:
	if (memStream) {
		// reset stream position 
		pgf.ResetStreamPos(false);
	} else {
		// close file
#ifdef WIN32
		if (fd) CloseHandle(fd);
#elif defined(__POSIX__)
		if (fd) close(fd);
#endif 
		delete stream;
	}

	if (returnValue) {
		// output: output file has to be closed before FileSize
		double destSize = (memStream) ? writtenBytes : double(FileSize(dest));
		double ratio = double(FileSize(source))/destSize;

		if (bQuiet) {
			cout << double(end - mean)/CLOCKS_PER_SEC << ';' << double(end - start)/CLOCKS_PER_SEC << ';' << ratio << ';';
		} else {
			cout << "Written bytes: " << writtenBytes << endl;
			cout << "Encoding time (encoding, writing PGF): " << double(end - mean)/CLOCKS_PER_SEC << " s" << endl;
			cout << "Total time (reading source, encoding, writing PGF): " << double(end - start)/CLOCKS_PER_SEC << " s" << endl;
			cout << "Compression ratio: " << ratio << endl;
			cout << endl;
		}
	}

	return returnValue;
}

/////////////////////////////////////////////////////////////////////////////
static bool Decoding(CImage*& image, char *source, char *dest, bool roi, PGFRect& rect, bool streaming, CPGFMemoryStream** memStream) {
#ifndef __PGFROISUPPORT__
	ASSERT(!roi);
	ASSERT(!streaming);
	rect;
#endif
	ASSERT(dest);
	bool returnValue = true;
	CPGFImage pgf;
	clock_t start = 0, mean = 0, mean2 = 0, mean3 = 0, end = 0;
	CPGFStream *stream = nullptr;
	UINT8* buff = nullptr;
	double sourceSize = 0;
#ifdef WIN32
	HANDLE fd = nullptr;
#elif defined(__POSIX__)
	int fd = 0;
#endif

	if (memStream) {
		// use memory stream
		stream = *memStream;
	} else {
		ASSERT(source);
	#ifdef WIN32
		// no other choice to get a "HANDLE"
		fd = CreateFile(source, GENERIC_READ, FILE_SHARE_READ, 0, OPEN_EXISTING, 0, 0);
		if (fd == INVALID_HANDLE_VALUE) {
			cerr << "Error: Could not open source file." << endl;
			fd = nullptr;
			returnValue = false;
			goto CleanUp;
		}

		// create stream object
		stream = new CPGFFileStream(fd);

	#elif defined(__POSIX__)
		fd = open(source, O_RDONLY);
		if (fd == -1){
			cout << "Error: Could not open source file." << endl;
			fd = 0;
			returnValue = false;
			goto CleanUp;
		}

		// create stream object
		stream = new CPGFFileStream(fd);
	#endif
	}

	// optional PGF encoder configuration
	pgf.ConfigureDecoder(true); // true: use openMP (if codec is compiled with openMP)

	if (memStream) {
		if (!bQuiet) cout << "Decoding PGF image from memory stream." << endl;
	} else {
		if (!bQuiet) cout << "Decoding PGF image: " << source << endl;
	}

	start = clock();
	try {
		// open pgf image
		UINT64 startpos = stream->GetPos();
		pgf.Open(stream);
		if (!bQuiet) cout << "Read header and level info [" << stream->GetPos() - startpos << " bytes]" << endl;

		// read annotations
		if (!bQuiet) {
			UINT32 len = 0;
			const UINT8 *data = pgf.GetUserData(len);
			if (data && len) {
				cout << (char *)data << endl;
			}
		}
		
		// read pgf image
	#ifdef __PGFROISUPPORT__
		if (pgf.ROIisSupported()) {
			if (memStream) {
				if (!bQuiet) cout << "PGF image supports ROI." << endl;
			} else {
				if (!bQuiet) cout << source << " supports ROI." << endl;
			}
		}
		if (roi) {
			pgf.Read(rect); // ROI test
			// example of how to make a second read (with a different ROI) after resetting stream position to data
			//pgf.ResetStreamPos(true);
			//pgf.Read(rect);
		} else {
			if (streaming) {
				// decode each level separately
				for (int i = pgf.Levels() - 1; i >= 0; i--) {
					if (!bQuiet) cout << "Read level " << i;
					UINT64 pos = stream->GetPos();
					pgf.Read(i);
					if (!bQuiet) cout << " [" << stream->GetPos() - pos << " bytes]" << endl;
				}
			}
		}
	#else
		cout << "PGF Console: ROI supports has been disabled." << endl;
	#endif
		if (!roi && !streaming) {
			// read entire file down to level 0
			pgf.Read();
		}
		sourceSize = double(stream->GetPos() - startpos);

	} catch(IOException& e) {
		int err = e.error;
		if (err >= AppError) {
			cerr << "Error: Opening and reading PGF image failed\n(" << PGFErrors[err - AppError] << ")!" << endl;
		} else {
			cerr << "Error: Opening and reading PGF image failed (" << err << ")!" << endl;
		}

		returnValue = false;
		goto CleanUp;
	}
	mean = clock();

	// create image
	image = new CImage();
#ifdef __PGFROISUPPORT__
	if (roi) {
		if (!image->Create(rect.Width(), rect.Height(), pgf.GetHeader()->mode)) {
			cerr << "Decoding is not supported for this PGF color format." << endl;
			returnValue = false;
			goto CleanUp;
		}
	} else
#endif
	{
		if (!image->Create(pgf.Width(), pgf.Height(), pgf.GetHeader()->mode)) {
			cerr << "Decoding is not supported for this PGF color format." << endl;
			returnValue = false;
			goto CleanUp;
		}
	}

	// copy bits
	mean2 = clock();
	buff = (UINT8 *)image->GetBits();
	if (pgf.Mode() == ImageModeRGB48) {
		int map[] = { 2, 1, 0 };
		pgf.GetBitmap(-image->GetPitch(), &(buff[(image->GetHeight() - 1) * image->GetPitch()]), image->GetBPP(), map);
	} else {
#ifdef __BIG_ENDIAN__
		int map[] = { 2, 1, 0 };
		pgf.GetBitmap(-image->GetPitch(), &(buff[(image->GetHeight() - 1) * image->GetPitch()]), image->GetBPP(), map);
#else
		pgf.GetBitmap(-image->GetPitch(), &(buff[(image->GetHeight() - 1) * image->GetPitch()]), image->GetBPP());
#endif
	}

	// update color table if image is indexed or bitmap
	if ((pgf.Mode() == ImageModeIndexedColor)) {
		// cannot get number of color table entries directly, so use 2^bitdepth
		image->SetColorTable(0, 1 << pgf.BPP(), pgf.GetColorTable());
	} else if (pgf.GetHeader()->mode == ImageModeBitmap) {
		RGBQUAD bw[2];
		bw[0].rgbRed = 255;
		bw[0].rgbGreen = 255;
		bw[0].rgbBlue = 255;
		bw[1].rgbRed = 0;
		bw[1].rgbGreen = 0;
		bw[1].rgbBlue = 0;
		image->SetColorTable(0, 2, bw);
	}

	mean3 = clock();

#ifdef __PNMEXSUPPORT__
	if (image->GetChannelDepth() > 8) {
		// set maximum value
		image->SetMaxValue(pgf.GetMaxValue());
	}
#endif

	// save image
	if (image->Save(dest)) {
		if (!bQuiet) cout << "Written image file: " << dest << endl;
	} else {
		cerr << "Error: Output format not supported for this color depth." << endl;
		return false;
	}

	end = clock();

CleanUp:
	if (memStream) {
		stream->SetPos(FSFromStart, 0);
	} else {
		// close file
#ifdef WIN32
		if (fd) CloseHandle(fd);
#elif defined(__POSIX__)
		if (fd) close(fd);
#endif 
		delete stream;
	}

	if (!memStream) {
		// source has to be closed before FileSize
		sourceSize = (double)FileSize(source);
	}

	if (returnValue) {
		// output
		double ratio = double(FileSize(dest))/sourceSize;
		if (bQuiet) {
			cout << double(mean3 - mean2 + mean - start)/CLOCKS_PER_SEC << ';' << double(end - mean3 + mean3 - mean2 + mean - start)/CLOCKS_PER_SEC << ';' << ratio << ';';
		} else {
			cout << "Decoding time (reading PGF, decoding): " << double(mean3 - mean2 + mean - start)/CLOCKS_PER_SEC << " s" << endl;
			cout << "Total time (reading PGF, decoding, writing destination): " << double(end - mean3 + mean3 - mean2 + mean - start)/CLOCKS_PER_SEC << " s" << endl;
			cout << "Compression ratio: " << ratio << endl;
			cout << endl;
		}
	}
	return returnValue;
}

/////////////////////////////////////////////////////////////////////////////
static bool Measurement(char *source, char *dest, char *temp, int levels, int quality, bool roi, PGFRect& rect, bool streaming, bool useMemStream) {
	ASSERT(source);
	ASSERT(dest);
	CImage *image1 = nullptr, *image2 = nullptr;
	CPGFMemoryStream *memStream = nullptr;

	if (!bQuiet) cout << "Measuring PSNR (quality: " << quality << ", levels: " << levels << "): " << source << endl;
	if (useMemStream) {
		if (!bQuiet) cout << "PGF image is written into memory stream." << endl;
	} else {
		ASSERT(temp);
	}
	if (!bQuiet) cout << endl;

	if (!Encoding(image1, source, temp, levels, quality, roi, streaming, (useMemStream) ? &memStream : nullptr)) {
		delete image1;
		delete memStream;
		return false;
	}
	if (!Decoding(image2, temp, dest, roi, rect, streaming, (useMemStream) ? &memStream : nullptr)) {
		delete image2;
		delete memStream;
		return false;
	}

	if (image1->GetChannelDepth() == 8 || image1->GetChannelDepth() == 16) {
		// output psnr
		PSNR(*image1, *image2, roi, rect);
	}

	delete image1;
	delete image2;
	delete memStream;
	return true;
}

/////////////////////////////////////////////////////////////////////////////
int main(int argc, char* argv[]) {
	int nRetCode = 0;

	// parse arguments
	enum Operation {Encode, Decode, Measure} op = Encode;
	char *source=nullptr, *dest=nullptr, *temp=nullptr;
	int levels = 0, quality = 0;
	PGFRect rect;
	int arg = 1;			// argument 0 = pgf
	bool bSource = true;	// true: bSource, false: dest
	bool bStreaming = false;// true: level wise writing/reading is used
	bool bROI = false;		// true: ROI is used
	bool bMemStream = false;// true: use a memory stream during measuring
	bool bWrongArgs = (argc < 4);

	while (!bWrongArgs && arg < argc) {
		if (argv[arg][0] == '-') {
			// options
			switch(argv[arg][1]) {
			case 'e': op = Encode; arg++; break;
			case 'd': op = Decode; arg++; break;
			case 'm': op = Measure;  
				if (argv[arg][2] == 'm') {
					// -mm
					bMemStream = true;
					arg++;
				} else {
					arg++;
					if (arg == argc) {
						bWrongArgs = true; 
					} else {
						temp = argv[arg]; arg++;
					}
				}
				break;
			case 'l': arg++; 
				if (arg == argc) {
					bWrongArgs = true; 
				} else {
					levels = atoi(argv[arg]); arg++; 
					bWrongArgs = (levels < 1) || (levels > MaxLevel);
				}
				break;
			case 'q': arg++; 
				if (arg == argc) {
					bWrongArgs = true; 
				} else {
					quality = atoi(argv[arg]); arg++;
					bWrongArgs = (quality < 0) || (quality > MaxQuality);
				}
				break;
		#ifdef __PGFROISUPPORT__
			case 'r': bROI = true;
				if (argv[arg][2] == 'e') {
					// -re[ct]
					arg++;
					if (arg + 4 > argc) {
						bWrongArgs = true;
					} else {
						rect = PGFRect(atoi(argv[arg]), atoi(argv[arg+1]), atoi(argv[arg+2]), atoi(argv[arg+3]));
						arg += 4;
					}
				} else {
					// -r
					arg++;
					if (arg == argc) bWrongArgs = true;
				}
				break;
			case 's': bStreaming = true; arg++; break;
		#endif // __PGFROISUPPORT__
			case 'v': bQuiet = true; arg++; break;
			default: arg++; bWrongArgs = true; break;
			}
		} else {
			if (bSource) {
				source = argv[arg];
				bSource = false;
			} else {
				dest = argv[arg];
			}
			arg++;
		}
	}
	if (!bQuiet) cout << "PGF Console - Copyright (c) 2001-" CurrentYear " xeraina GmbH, Switzerland" << endl <<
		"Console Version: " PGFConsoleVersion ", www.xeraina.ch" << endl <<
		"libpgf Version : " PGFCodecVersion ", www.libpgf.org" << endl << endl;
	if (bWrongArgs) {
		if (!bQuiet) {
			cout << "Usage: " << endl <<
			#ifdef __PGFROISUPPORT__
					"- Encoding: pgfconsole -e [-l levels] [-q quality] [-r] [-s] [-v] source dest" << endl <<
			#else
					"- Encoding: pgfconsole -e [-l levels] [-q quality] [-v] source dest" << endl <<
			#endif
					"               Create from a source file a PGF image (dest)." << endl << 
					"               The most popular image file formats with the following image" << endl <<
					"               types are supported:" << endl <<
					"               - bitmap (1 bit)" << endl <<
					"               - grayscale (8 and 16 bit)" << endl <<
					"               - indexed color (8 bit)" << endl <<
					"               - RGB (16 [565], 24, 32, and 48 bit)" << endl <<
					"               - RGBA (32 bit)" << endl <<
					"  Options:" << endl << 
					"  -l levels    Number of hierarchical levels [1.." << MaxLevel << "]. Default is 0." << endl <<
					"               0 means the number of levels are automatically set." << endl <<
					"  -q quality   Quality [0.." << MaxQuality << "]. Default is 0." << endl << 
					"               0 means perfect quality (lossless compression)," << endl << 
					"               " << MaxQuality << " means maximum compression." << endl << 
			#ifdef __PGFROISUPPORT__
					"  -r           Region of interest (ROI) encoding scheme is used." << endl <<
					"               This encoding scheme has a slightly worse compression ratio." << endl <<
					"  -s           Level wise encoding in separate writing calls." << endl <<
			#endif
					"  -v           Numbers only: All text output is reduced to numbers." << endl << 
					endl << 
			#ifdef __PGFROISUPPORT__
					"- Decoding: pgfconsole -d [-rect left top width height] [-s] [-v] source dest" << endl <<
			#else
					"- Decoding: pgfconsole -d [-v] source dest" << endl <<
			#endif
					"               Create from a PGF image (source) a new image (dest)." << endl << 
					"  Options:" << endl << 
			#ifdef __PGFROISUPPORT__
				"  -rect rect   Read a rectangular region of a PGF image supporting Region of" << endl <<
					"               interests (ROI). The rectangle is defined by 4 blank-separated" << endl << 
					"               positive parameters: left top width height" << endl <<
					"  -s           Level wise decoding in separate reading calls." << endl <<
			#endif
					"  -v           Numbers only: All text output is reduced to numbers." << endl << 
					endl << 
				    "- Measuring: pgfconsole -m temp-file [...] source destination" << endl << 
					"               Measure quality between source and destination bitmap." << endl <<
					"               Encode from an input image (source) a PGF image" << endl << 
					"               (temp-file) and decode from the temp-file a new output" << endl << 
					"               (destination image)." << endl << 
					"  Options:" << endl <<
					"  -mm          Instead of using the option -m temp-file you can use " << endl << 
					"               the option -mm (without temp-file). The latter writes the PGF" << endl <<
					"               image into a memory stream instead of a file stream." << endl <<
					"               In both cases all encoding and decoding options are valid." << endl << 
					endl << 
					endl;
		}
		nRetCode = 2;
	} else {
		CImage *image = 0;

#ifdef __PNMEXSUPPORT__
		// register new PNM plugin
		CImage::RegisterPNM();
#endif

		switch(op) {
		case Encode:
			if (!Encoding(image, source, dest, levels, quality, bROI, bStreaming, nullptr)) nRetCode = 3;
			break;
		case Decode:
			if (!Decoding(image, source, dest, bROI, rect, bStreaming, nullptr)) nRetCode = 4;
			break;
		case Measure:
			if (!Measurement(source, dest, temp, levels, quality, bROI, rect, bStreaming, bMemStream)) nRetCode = 5;
			break;
		default:
			nRetCode = 6;
			ASSERT(false); // unhandled operation
		}
		if (bQuiet) cout << endl;

		delete image;
	}
	return nRetCode;
}

