// Copyright (c) 2023 Yawning Angel
//
// SPDX-License-Identifier: BSD-3-Clause

package helpers

import (
	"encoding/binary"
	"encoding/hex"
	"math/bits"
	"strings"
)

// Uint64IsZero returns 1 iff `u == 0`, 0 otherwise, in constant time.
func Uint64IsZero(u uint64) uint64 {
	isNonzero := Uint64IsNonzero(u)
	return (^isNonzero) & 1
}

// Uint64IsNonzero returns 1 iff `u != 0`, 0 otherwise, in constant time.
func Uint64IsNonzero(u uint64) uint64 {
	_, borrow := bits.Sub64(0, u, 0)
	return borrow
}

// Uint64Equal returns 1 iff `a == b`, 0 otherwise, in constant time.
func Uint64Equal(a, b uint64) uint64 {
	return Uint64IsZero(a ^ b)
}

// FiatLimbsAreEqual returns 1 iff `a == b`, 0 otherwise, in constant time.
// This is correct for the autogenerated fiat routines, as values are
// always in the range `0 <= n < m` (ie: fully reduced).
func FiatLimbsAreEqual(a, b *[4]uint64) uint64 {
	var v uint64

	for i := 0; i < len(a); i++ {
		v |= a[i] ^ b[i]
	}

	return Uint64IsZero(v)
}

// BytesToSaturated interprets src as a 256-bit big-endian integer, and
// returns the 64-bit saturated representation, compatible with the
// autogenerated fiat routines.
func BytesToSaturated(src *[32]byte) [4]uint64 {
	var dst [4]uint64
	dst[3] = binary.BigEndian.Uint64(src[0:8])
	dst[2] = binary.BigEndian.Uint64(src[8:16])
	dst[1] = binary.BigEndian.Uint64(src[16:24])
	dst[0] = binary.BigEndian.Uint64(src[24:32])
	return dst
}

// PutSaturatedToBytes fills dst with the 256-big big-endian byte
// encoding of src, where src is the fiat-compatible 64-bit sautrated
// representation, and returns dst.
func PutSaturatedToBytes(dst *[32]byte, src *[4]uint64) []byte {
	binary.BigEndian.PutUint64(dst[0:8], src[3])
	binary.BigEndian.PutUint64(dst[8:16], src[2])
	binary.BigEndian.PutUint64(dst[16:24], src[1])
	binary.BigEndian.PutUint64(dst[24:32], src[0])
	return dst[:]
}

// MustBytesFromHex interprets src as a hex encoded byte-slice, and
// returns the byte representation, or panics.  `src` MAY be prefixed
// with `0x`.
func MustBytesFromHex(src string) []byte {
	src = TrimOhEcks(src)

	b, err := hex.DecodeString(src)
	if err != nil {
		panic("internal/helpers: invalid hex string: " + err.Error())
	}

	return b
}

// Must256BitsFromHex interprets src as a hex encoded big-endian integer
// that is less than 2^256, or panics.  `src` MUST be less than or
// equal to 64-bytes in length after the optional "0x" prefix is removed.
func Must256BitsFromHex(src string) *[32]byte {
	src = TrimOhEcks(src)

	padSz := 32*2 - len(src)
	switch {
	case padSz > 0:
		src = strings.Repeat("0", padSz) + src
	case padSz < 0:
		panic("internal/helpers: over sized hex string")
	}
	b := MustBytesFromHex(src)

	return (*[32]byte)(b)
}

// TrimOhEcks trims the "0x" hex prefix if present.
func TrimOhEcks(s string) string {
	return strings.TrimPrefix(s, "0x")
}
